#include "StdAfx.h"

#include "scripting/ScriptAggregateParam.h"
#include "scripting/MScriptInterp.h"
#include "scripting/ScriptUtils.h"
#include <scripting/ScriptVector3.h>

#include <params/MAggregateParameter.h>

#include "jscntxt.h"

#include <assert.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {  

class AggregateEnumState {
public:
	int numParams;	// Total # of parameters to iterate over
	int index;		// Current property index
};

class AztecAggregateParam {
public:
	AztecAggregateParam() {
		parent = NULL;
		param = NULL;
	}
	MNamedObjectPtr parent;
	MAggregateParameterPtr param;
};

static JSBool
	aztecAggregateLength(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecAggregateParam *param = (AztecAggregateParam *)JS_GetPrivate(cx, obj);
	if (param == NULL || param->param == NULL) {
		return JS_FALSE;
	}

	// This method takes no parameters, so report an aerror if we are given any.
	if (argc != 0) {
		return JS_FALSE;
	}

	*rval = INT_TO_JSVAL(param->param->getMemberCount());
	return JS_TRUE;
}


static JSPropertySpec aztecAggregate_props[] = {
	//	{"x",	0,	JSPROP_ENUMERATE},
	//	{"y",	1,	JSPROP_ENUMERATE},
	//	{"z",	2,	JSPROP_ENUMERATE},
	{0}
};

static JSFunctionSpec aztecAggregate_methods[] = {
	{"size",	aztecAggregateLength,	0},
	{0}
};
// See if a particular property exists in the Aztec object.  
static JSBool
	aztec_lookupProperty(JSContext *cx, JSObject *obj, jsid id,
	JSObject **objp, JSProperty **propp)
{
	AztecAggregateParam *param = (AztecAggregateParam*)JS_GetPrivate(cx, obj);
	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
	if (param == NULL || interp == NULL) {
		return JS_TRUE;
	}

	// OutputDebugString("Aztec lookupProperty\n");

	jsval idval;
	if (!JS_IdToValue(cx, id, &idval)) {
		// Failed to resolve the property name
		*objp = NULL;
		*propp = NULL;
		return JS_TRUE;
	}

	if (JSVAL_IS_STRING(idval)) {
		// Look for a property by name
		char *prop_name = JS_GetStringBytes(JSVAL_TO_STRING(idval));

		// See if any of the properties on the list match prop_name
		int index = param->param->getMemberIndex(prop_name);
		if (index >= 0) {
			*objp  = obj;
			*propp = (JSProperty *)1;
			return JS_TRUE;
		}

		// TODO assign a proper value to this. 
		*propp = NULL;

	} else if (JSVAL_IS_INT(idval)) {
		// Look for a property by index
		int index = JSVAL_TO_INT(idval);

		*objp = obj;

		// TODO assign a proper value to this. 
		*propp = NULL;

	} else {
		// Failed to resolve the property name
		*objp = NULL;
		*propp = NULL;
	}

	return JS_TRUE;
}

static JSBool
	aztec_defineProperty(JSContext *cx, JSObject *obj, jsid id, jsval value,
	JSPropertyOp getter, JSPropertyOp setter,
	uintN attrs, JSProperty **propp)
{
	// OutputDebugString("Attempt to define aztec property\n");

	return JS_TRUE;
}

static JSBool
	aztec_getPropertyById(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	jsval idval;
	if (!JS_IdToValue(cx, id, &idval)) {
		return JS_FALSE;
	}

	AztecAggregateParam *param = (AztecAggregateParam *)JS_GetPrivate(cx, obj);
	if (param == NULL || param->param == NULL) {
		return JS_FALSE;
	}

	// If we are working from a parameter, then pull the current value
	// and use it.  Otherwise used the Aggregate we are dragging around
	// inside this object.

	if (JSVAL_IS_STRING(idval)) {
		// Look for a property by name
		char *prop_name = JS_GetStringBytes(JSVAL_TO_STRING(idval));
		// See if any of the properties on the list match prop_name
		int index = param->param->getMemberIndex(prop_name);

		if (index >= 0) {
			// Found a member of the aggregate parameter that matches the input string.
			MParameterObjectPtr paramPtr = param->param->getMember(index);

			// If the parameter object said we have a parameter with that index, then it
			// really needs to give us one when we ask.
			assert(paramPtr != NULL);

			// Turn this piece of the parameter into a scriptable object...
			*vp = convertParam(cx, obj, param->parent, paramPtr);

			return JS_TRUE;
		}

		// May be a built-in method.
		for (int m=0;aztecAggregate_methods[m].name != NULL;m++) {
			if (!strcmp(prop_name, aztecAggregate_methods[m].name)) {
				// Yup - is a built-in function.
				JSFunction *jfunc =
					JS_NewFunction(cx, aztecAggregate_methods[m].call,
					aztecAggregate_methods[m].nargs,
					0, obj, prop_name);
				JSObject *jfobj = JS_GetFunctionObject(jfunc);
				*vp = OBJECT_TO_JSVAL(jfobj);
				return JS_TRUE;
			}
		}

	} else if (JSVAL_IS_INT(idval)) {
		int index = JSVAL_TO_INT(idval);

		*vp = convertParam(cx, obj, param->parent, param->param->getMember(index));

		return JS_TRUE;

	} else if (JSVAL_IS_OBJECT(idval)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, idval));
		OutputDebugString("Aggregate prop: ");
		OutputDebugString(propStr);
		OutputDebugString("\n");
		return JS_FALSE;
	} else {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, idval));
		OutputDebugString("Aggregate prop: ");
		OutputDebugString(propStr);
		OutputDebugString("\n");
		return JS_FALSE;
	}

	return JS_FALSE;
}

static JSBool
	aztec_setPropertyById(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	jsval idval;
	if (!JS_IdToValue(cx, id, &idval)) {
		return JS_FALSE;
	}

	AztecAggregateParam *param = (AztecAggregateParam *)JS_GetPrivate(cx, obj);
	if (param == NULL || param->param == NULL) {
		return JS_FALSE;
	}

	// make sure our index parameter is of an integer type
	int index = -1;

	if (JSVAL_IS_STRING(idval)) {
		// Look for a property by name
		char *prop_name = JS_GetStringBytes(JSVAL_TO_STRING(idval));
		// See if any of the properties on the list match prop_name
		index = param->param->getMemberIndex(prop_name);

	} else if (JSVAL_IS_INT(idval)) {
		index = JSVAL_TO_INT(idval);
	}

	if (index != -1) {
		if (JSVAL_IS_STRING(*vp)) {
			// Look for a property by name
			char *value = JS_GetStringBytes(JSVAL_TO_STRING(*vp));

			param->param->getMember(index)->setValueString(MStr(value));

		} 

		if (JSVAL_IS_OBJECT(*vp)) {
			// This is an object param, so it is ok to hand it an object.
			JSObject *jsobj = JSVAL_TO_OBJECT(*vp);

			// attempt to convert it to a vector3 class.
			void *priv = JS_GetInstancePrivate(cx, jsobj, &aztecVector3_class, NULL);

			// if we have a successful convesion, then set the value appropriately
			if (priv != NULL) {
				MVector3 *nvec = (MVector3 *)priv;
				param->param->getMember(index)->setValueVector(*nvec);
				return JS_TRUE;
			}
		}


	} else if (JSVAL_IS_OBJECT(idval)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, idval));
		OutputDebugString("Aggregate prop: ");
		OutputDebugString(propStr);
		OutputDebugString("\n");
		return JS_FALSE;
	} else {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, idval));
		OutputDebugString("Aggregate prop: ");
		OutputDebugString(propStr);
		OutputDebugString("\n");
		return JS_FALSE;
	}

	return JS_FALSE;
}

static JSBool
	aztec_getAttributes(JSContext *cx, JSObject *obj, jsid id,
	JSProperty *prop, uintN *attrsp)
{
	OutputDebugString("Aztec getAttributes\n");

	// We don't maintain JS property attributes for Aztec objects
	*attrsp = JSPROP_PERMANENT | JSPROP_ENUMERATE;
	return JS_FALSE;
}

static JSBool
	aztec_setAttributes(JSContext *cx, JSObject *obj, jsid id,
	JSProperty *prop, uintN *attrsp)
{
	OutputDebugString("Aztec setAttributes\n");

	// We don't maintain JS property attributes for Aztec objects
	if (*attrsp != (JSPROP_PERMANENT | JSPROP_ENUMERATE)) {
		return JS_FALSE;
	}

	// Silently ignore all setAttribute attempts
	return JS_TRUE;
}

static JSBool
	aztec_deleteProperty(JSContext *cx, JSObject *obj, jsid id, jsval *vp)
{
	OutputDebugString("Aztec deleteProperty\n");

	return JS_FALSE;
}

static JSBool
	aztec_defaultValue(JSContext *cx, JSObject *obj, JSType typ, jsval *vp)
{
	// OutputDebugString("Aztec defaultValue\n");
	switch (typ) {
case JSTYPE_OBJECT:
	*vp = OBJECT_TO_JSVAL(obj);
	return JS_TRUE;

case JSTYPE_FUNCTION:
	return JS_FALSE;

case JSTYPE_VOID:
case JSTYPE_STRING:
	{
		AztecAggregateParam *param = (AztecAggregateParam *)JS_GetPrivate(cx, obj);
		if (param == NULL) {
			JSString *str = JS_NewStringCopyZ(cx, "[Aztec Object - noparam]");
			*vp = STRING_TO_JSVAL(str);
		} else {
			MStr valStr;
			valStr = "[Aztec Aggregate]";
			JSString *str = JS_NewStringCopyZ(cx, valStr);
			*vp = STRING_TO_JSVAL(str);
		}
		return JS_TRUE;
	}

case JSTYPE_NUMBER:
	return JS_FALSE;

case JSTYPE_BOOLEAN:
	return JS_FALSE;

default:
	return JS_FALSE;
	}
}

static JSBool
	aztec_newEnumerate(JSContext *cx, JSObject *obj, JSIterateOp enum_op,
	jsval *statep, jsid *idp)
{
	AztecAggregateParam *param = (AztecAggregateParam *)JS_GetPrivate(cx, obj);

	// Check for native object
	if (param == NULL) {
		// No native object - say we don't have any properties.
		*statep = JSVAL_NULL;
		if (idp != NULL) {
			*idp = INT_TO_JSVAL(0);
		}
		return JS_TRUE;
	}

	switch(enum_op) {
case JSENUMERATE_INIT:
	{
		int paramCount = param->param->getMemberCount();

		AggregateEnumState *enumState = new AggregateEnumState;
		enumState->numParams = paramCount;
		enumState->index = 0;
		*statep = PRIVATE_TO_JSVAL(enumState);
		if (idp != NULL) {
			*idp = INT_TO_JSVAL(paramCount);
		}
		return JS_TRUE;
	}

case JSENUMERATE_NEXT:
	{
		AggregateEnumState *enumState = (AggregateEnumState *)JSVAL_TO_PRIVATE(*statep);

		if (enumState->index < enumState->numParams) {
			MStr paramName = param->param->getMemberName(enumState->index);
			JSString *str = JS_NewStringCopyZ(cx, paramName.c_str());

			JS_ValueToId(cx, STRING_TO_JSVAL(str), idp);
			enumState->index++;
			return JS_TRUE;
		}
		// Drop through to destroy
	}

case JSENUMERATE_DESTROY:
	{
		AggregateEnumState *enumState = (AggregateEnumState *)JSVAL_TO_PRIVATE(*statep);
		delete enumState;
		*statep = JSVAL_NULL;
		return JS_TRUE;
	}

default:
	OutputDebugString("Bad enumeration state\n");
	return JS_FALSE;
	}
}

static JSBool
	aztec_checkAccess(JSContext *cx, JSObject *obj, jsid id,
	JSAccessMode mode, jsval *vp, uintN *attrsp)
{
	OutputDebugString("Aztec Aggregate checkAccess\n");

	switch (mode) {
case JSACC_WATCH:
	return JS_FALSE;

case JSACC_IMPORT:
	return JS_FALSE;

default:
	return JS_TRUE;
	}
}


static JSBool
	aztecAggregate_convert(JSContext *cx, JSObject *obj, JSType typ, jsval *vp)
{
	AztecAggregateParam *param = (AztecAggregateParam *)JS_GetPrivate(cx, obj);
	if (param == NULL || param->param == NULL) {
		return JS_FALSE;
	}

	switch (typ) {
case JSTYPE_OBJECT:
	*vp = OBJECT_TO_JSVAL(obj);
	return JS_TRUE;

case JSTYPE_VOID:
case JSTYPE_STRING:
	{
		char buf[128];
		sprintf(buf, "aggregate");
		JSString *str = JS_NewStringCopyZ(cx, buf);
		*vp = STRING_TO_JSVAL(str);
		return JS_TRUE;
	}

case JSTYPE_FUNCTION:
case JSTYPE_NUMBER:
case JSTYPE_BOOLEAN:
default:
	return JS_FALSE;
	}
}

static void
	aztecAggregate_finalize(JSContext *cx, JSObject *obj)
{
	AztecAggregateParam *param = (AztecAggregateParam *)JS_GetPrivate(cx, obj);
	if (param == NULL) {
		return;
	}

	delete param;
}

// Specialized object operations - these allow us to dynamically reflect
// Aztec properties into the script interpreter.
JSObjectOps aztecAggregate_ops = {
	// Mandatory non-null function pointer members.
	js_ObjectOps.newObjectMap,
		js_ObjectOps.destroyObjectMap,
		aztec_lookupProperty,
		aztec_defineProperty,
		aztec_getPropertyById,  // getProperty
		aztec_setPropertyById,  // setProperty
		aztec_getAttributes,
		aztec_setAttributes,
		aztec_deleteProperty,
		aztec_defaultValue,
		aztec_newEnumerate,
		aztec_checkAccess,

		// Optionally non-null members start here.
		NULL,                       /* thisObject */
		NULL,                       /* dropProperty */
		NULL,                       /* call */
		NULL,                       /* construct */
		NULL,                       /* xdrObject */
		NULL,                       /* hasInstance */
		NULL,                       /* setProto */
		NULL,                       /* setParent */
		NULL,                       /* mark */
		NULL,                       /* clear */
		NULL,                       /* getRequiredSlot */
		NULL                        /* setRequiredSlot */
};


static JSObjectOps *
	aztecAggregate_getObjectOps(JSContext *cx, JSClass *clazz)
{
	return &aztecAggregate_ops;
}

JSClass aztecAggregate_class = { 
	"AztecAggregate", JSCLASS_HAS_PRIVATE, 
		JS_PropertyStub, JS_PropertyStub, // add/del property
		JS_PropertyStub, JS_PropertyStub, // get/set property
		JS_EnumerateStub, JS_ResolveStub,
		aztecAggregate_convert, aztecAggregate_finalize,
		aztecAggregate_getObjectOps
};

JSObject *newAggregateParam(JSContext *cx, 
	const MNamedObjectPtr &aztec_obj,
	const MAggregateParameterPtr &aztec_param)
{
	JSObject *jsvec = JS_NewObject(cx, &aztecAggregate_class, NULL, JS_GetGlobalObject(cx));
	AztecAggregateParam *vparm = new AztecAggregateParam;
	vparm->parent = aztec_obj;
	vparm->param = aztec_param;
	JS_SetPrivate(cx, jsvec, vparm);

	return jsvec;
}


void addAggregateParamClass(JSContext *cx)
{
	// Add a vector class (reflection of MVector3) to the global namespace
	JS_InitClass(cx, JS_GetGlobalObject(cx), NULL,
		&aztecAggregate_class, 0, 0,
		aztecAggregate_props, aztecAggregate_methods, 
		0, 
		0);

}

}
