#include <StdAfx.h>

// disable 'debug name is longer than 255 chars'
#pragma warning(disable : 4786)

#include <fstream>
#include <stdlib.h>

#ifdef WIN32
  #include <io.h>
#else
  #include <dirent.h>
  #include <dlfcn.h>
  #include <unistd.h>
  #include <sys/param.h>

  #define _access    access
  #define _getcwd    getcwd
  #define _MAX_PATH  MAXPATHLEN
#endif

#include "scripting/MScriptInterp.h"
#include "scripting/ScriptScene.h"
#include "scripting/ScriptSceneObject.h"
#include "scripting/ScriptVectorParam.h"
#include "scripting/ScriptUtils.h"
#include "scripting/ScriptVector3.h"
#include "scripting/ScriptMatrix4.h"
#include "scripting/ScriptRGBAFloat.h"
#include "scripting/ScriptVertex.h"
#include "scripting/ScriptTriangle.h"
#include "scripting/ScriptMesh.h"
#include "scripting/ScriptArrayParam.h"

#include <misc/ResourceManager.h>

extern "C" {
#include "jsstddef.h"
#include "jscntxt.h"
#include "jsprf.h"
#include "jsscope.h"
}

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

static JSBool reportWarnings = JS_TRUE;
static std::string scriptCurrentDir;


static void
aztec_ErrorReporter(JSContext *cx, const char *message, JSErrorReport *report)
{
	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);

    int i, j, k, n;
    char *prefix, *tmp;

    if (!report) {
        printf("%s\n", message);
        return;
    }

    /* Conditionally ignore reported warnings. */
    if (JSREPORT_IS_WARNING(report->flags) && !reportWarnings)
        return;

    prefix = NULL;
    if (report->filename)
        prefix = JS_smprintf("%s:", report->filename);
    if (report->lineno) {
        tmp = prefix;
        prefix = JS_smprintf("%s%u: ", tmp ? tmp : "", report->lineno);
        JS_free(cx, tmp);
    }
    if (JSREPORT_IS_WARNING(report->flags)) {
        tmp = prefix;
        prefix = JS_smprintf("%s%swarning: ",
                             tmp ? tmp : "",
                             JSREPORT_IS_STRICT(report->flags) ? "strict " : "");
        JS_free(cx, tmp);
    }

#if 0
    const char *ctmp;
    /* embedded newlines -- argh! */
    while ((ctmp = strchr(message, '\n')) != 0) {
        ctmp++;
        if (prefix)
            interp->PutStr(prefix);
        fwrite(message, 1, ctmp - message, gErrFile);
        message = ctmp;
    }
#endif

    /* If there were no filename or lineno, the prefix might be empty */
    if (prefix)
        interp->PutStr(prefix);
    interp->PutStr(message);

    if (!report->linebuf) {
		interp->PutStr("\n");
        goto out;
    }

    /* report->linebuf usually ends with a newline. */
    n = strlen(report->linebuf);
#if 1
	interp->PutStr(":\n");
	interp->PutStr(prefix);
	interp->PutStr(report->linebuf);
	interp->PutStr((n > 0 && report->linebuf[n-1] == '\n') ? "" : "\n");
	interp->PutStr(prefix);
#else
    fprintf(gErrFile, ":\n%s%s%s%s",
            prefix,
            report->linebuf,
            (n > 0 && report->linebuf[n-1] == '\n') ? "" : "\n",
            prefix);
#endif
    n = PTRDIFF(report->tokenptr, report->linebuf, char);
    for (i = j = 0; i < n; i++) {
        if (report->linebuf[i] == '\t') {
            for (k = (j + 8) & ~7; j < k; j++) {
                interp->PutStr(".");
            }
            continue;
        }
        interp->PutStr(".");
        j++;
    }
    interp->PutStr("^\n");
 out:
    JS_free(cx, prefix);
}

static JSBool
global_getProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
#if 0
    OutputDebugString("get global property ");
	OutputDebugString(JS_GetStringBytes(JS_ValueToString(cx, id)));
	OutputDebugString("\n");
	OutputDebugString("  current val: ");
	OutputDebugString(JS_GetStringBytes(JS_ValueToString(cx, *vp)));
	OutputDebugString("\n");
#endif

#if 0
	// Scene variables are now reflected throught the global "Scene"
	char *objStr = JS_GetStringBytes(JS_ValueToString(cx, id));

	MBaseObjectPtr Obj = findNamedObject(objStr);
	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
	if (Obj == NULL || interp == NULL) {
		return JS_TRUE;
	}

	JSObject *jobj = JS_NewObject(cx, &aztec_class, NULL, interp->GetGlobalObject());
	if (jobj != NULL) {
		JS_SetPrivate(cx, jobj, Obj.m_Ptr);
	}
	*vp = OBJECT_TO_JSVAL(jobj);
#endif

    return JS_TRUE;
}

static JSBool
global_resolve(JSContext *cx, JSObject *obj, jsval id)
{
#if 0
	// Scene objects are now reflected throught the global "Scene"
	char *objStr = JS_GetStringBytes(JS_ValueToString(cx, id));

	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
	MBaseObjectPtr Obj = findNamedObject(objStr);
	if (Obj != NULL || interp == NULL) {
		// It's not an Aztec object - just ignore.
		return JS_TRUE;
	}

	JSObject *jobj = JS_DefineObject(cx, interp->GetGlobalObject(),
		objStr, &aztec_class, 0, JSPROP_ENUMERATE);
	if (jobj != NULL) {
		JS_SetPrivate(cx, jobj, Obj.m_Ptr);
	}
#endif

    return JS_TRUE;
}

static void
aztec_LoadErrorReporter(JSContext *cx, const char *message, JSErrorReport *report)
{
	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);

    if (!report) {
        interp->PutStr(message);
        return;
    }

    /* Ignore any exceptions */
    if (JSREPORT_IS_EXCEPTION(report->flags)) {
        return;
	}

    /* Otherwise, fall back to the ordinary error reporter. */
    aztec_ErrorReporter(cx, message, report);
}

static bool
tryLoad(JSContext *cx, JSObject *obj, const char *filename)
{
	bool ok;
    errno = 0;

	// See if file exists
	if (_access(filename, 0) == -1) {
#if 0
		OutputDebugString("Can't find file: ");
		OutputDebugString(filename);
		OutputDebugString("\n");
#endif
		return false;
	}

	// File exists, try loading and compiling it's contents as a script.
    // JSErrorReporter older = JS_SetErrorReporter(cx, aztec_LoadErrorReporter);
    JSScript *script = JS_CompileFile(cx, obj, filename);
    if (!script) {
		OutputDebugString("Failed to compile file: ");
		OutputDebugString(filename);
		OutputDebugString("\n");
        ok = false;
    } else {
		jsval result;
        ok = (JS_ExecuteScript(cx, obj, script, &result) == JS_TRUE ? true : false);
		if (!ok) {
			OutputDebugString("Failed to execute script in file: ");
			OutputDebugString(filename);
			OutputDebugString("\n");
		}
        JS_DestroyScript(cx, script);
    }

	if (ok) {
		MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);

		char str[1024];
		sprintf(str, "Succesfully loaded script '%s'\n", filename);
		interp->PutStr(str);
	}

	// JS_SetErrorReporter(cx, older);

	return ok;
}

static JSBool
Load(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
    uintN i;
    JSString *str;
    const char *filename;

    // go thorugh each argument and try to load each as a script.
    for (i = 0; i < argc; i++) {
      // exrtact the string value from the jsval
      str = JS_ValueToString(cx, argv[i]);
      if (!str) {
        return JS_FALSE;
      }

      argv[i] = STRING_TO_JSVAL(str);
      filename = JS_GetStringBytes(str);

      // try to load the script relative to the current script directory.
      std::string path;

      if (path.length() == 0) {
        path = ResourceManager::locateResource(scriptCurrentDir + '/' + filename);
      }

      // try to locate this resource in the resource paths.
      if (path.length() == 0) {
        path = ResourceManager::locateResource(filename);
      }

      // if we couldn't find thes script, try for one in the scripts directory.
      if (path.length() == 0) {
        path = ResourceManager::locateResource(std::string("scripts/") + filename);
      }
      
      // try and load the script, report an error otherwise.
      if (path.length() != 0 && tryLoad(cx, obj, path.c_str())) {
        continue;
      } else {
      
        MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
        
        char str[1024];
        sprintf(str, "Failed to load script '%s'\n", filename);
        interp->PutStr(str);
      }
    }
    
    
    return JS_TRUE;
}


static JSBool
Print(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
    uintN i, n;
    JSString *str;

    for (i = n = 0; i < argc; i++) {
        str = JS_ValueToString(cx, argv[i]);
        if (!str)
            return JS_FALSE;
		interp->PutStr(JS_GetStringBytes(str));
    }
    n++;
    if (n) {
		interp->PutStr("\n");
	}
    return JS_TRUE;
}

// invoke the garbage collector.
static JSBool
GC(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval* rval){
	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
    JSRuntime *rt;
    uint32 preBytes;

    rt = cx->runtime;
    preBytes = rt->gcBytes;
    js_ForceGC(cx);
	char sbuf[128];
    sprintf(sbuf, "before %lu, after %lu\n",
            (unsigned long)preBytes, (unsigned long)rt->gcBytes);
	interp->PutStr(sbuf);

#ifdef JS_GCMETER
    js_DumpGCStats(rt, stdout);
#endif
    return JS_TRUE;

    JS_GC(cx);
    return JS_TRUE;
}

static JSBool
CD(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
  if (argc == 0) {
    // if we have no arguments, then we are just getting our current working dir
    *rval = STRING_TO_JSVAL( JS_NewStringCopyZ(cx, scriptCurrentDir.c_str()) );
    return JS_TRUE;
  }

  // otherwise we are changing the current dir.

  std::string newDir = JS_GetStringBytes( JS_ValueToString(cx, argv[0]) );

  // we can't do anything if nothing changes.
  if (newDir.length() == 0) {
    return JS_FALSE;
  }

  // first check to see if this is an absolute change (i.e. not relative to our current dir)
#ifdef _WIN32
  // if the second char is a colon, or we start with a slash, then its an absolute change
  if (newDir.size() >= 2 && newDir[1] == ':') {
    scriptCurrentDir = newDir;
    return JS_TRUE;
  } else if (newDir.size() >= 1 && newDir[0] == '/') {
    // we are trying to change to the root of the current drive. Check to see 
    // if we have a current drive, and then ust that.
    if (scriptCurrentDir.find(':') == 1) {
      scriptCurrentDir = scriptCurrentDir.substr(0, 2) + newDir;
      return JS_TRUE;
    } else if (scriptCurrentDir.find("//") == 0) {
      // we have a UNC name, so grab the first chunk before the slash and use it.
      int slashPos = scriptCurrentDir.find('/', 2);
      if (slashPos != std::string::npos) {
        scriptCurrentDir = scriptCurrentDir.substr(0, slashPos);
      }
      scriptCurrentDir += newDir;
      return JS_TRUE;
    }
  }

#else
  // if we are starting with a slash, then its an absolute change.
  if (newDir[0] == '/') {
    scriptCurrentDir = newDir;
    return JS_TRUE;
  }

#endif

  // if we get here then it is a simple concatenation of directories

  scriptCurrentDir += "/";
  scriptCurrentDir += newDir;

  // make sure that we DO NOT end in a slash.
  if (scriptCurrentDir[scriptCurrentDir.length() - 1] == '/') {
    scriptCurrentDir.substr(0, scriptCurrentDir.length() - 1);
  }

  return JS_TRUE;

}

static JSBool
LS(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	// Get the directory where the scripts are located
  std::string directory = scriptCurrentDir + '/';

    // Get a list of the filenames
	typedef std::vector<std::string> FilenameList;
    FilenameList filenames;
	MStr FileMask = "*.*";

#ifdef WIN32
    _finddata_t    FindData;
    int            hFind, Result;

    MStr actualSearchMask = directory + "\\" + FileMask;
    hFind = _findfirst((LPCTSTR)actualSearchMask, &FindData);
    
    if (hFind != -1) {
      Result = 0;
    } else {
      Result = 1;
    }
    
    while (Result == 0)
    {
      filenames.push_back(FindData.name);
      Result = _findnext(hFind, &FindData);
    }
    
    _findclose(hFind);
#else 
    MStr expr = FileMask;
    expr.RemoveLeading('*');
    expr.RemoveTrailing('*');

    DIR *dirStream = opendir(directory.c_str());
    struct dirent *dir;

    dir = readdir(dirStream);

    while (dir != NULL) {
      MStr filename(dir->d_name);
      if (filename.findSubstring(expr) != -1) {
        filenames.push_back(filename.c_str());
      }

      dir = readdir(dirStream);
    }

    closedir(dirStream);

#endif

	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
    FilenameList::iterator it;
    for (it = filenames.begin(); it != filenames.end(); ++it) {
		interp->PutStr((*it).c_str());
		interp->PutStr("\n");
      }
    
    return JS_TRUE;
}

static JSBool
Cat(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval) 
{
	// Get the full filename including path
	JSString *str;
	const char *filename;
	str = JS_ValueToString(cx, argv[0]);
	filename = JS_GetStringBytes(str);
		
	MStr fullpath;

	char buf[_MAX_PATH];
#if _WIN32
	GetModuleFileName(NULL, buf, _MAX_PATH);
	char drive[_MAX_DRIVE];
	char dir[_MAX_DIR];
	_splitpath(buf, drive, dir, NULL, NULL);
	fullpath = drive;
	fullpath += dir;
#else
	fullpath = _getcwd(buf, _MAX_PATH);
#endif
	fullpath += "../../common/Scripts/";
	fullpath += filename;


	// Dump the file
	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
	FILE *f;
	char *tbuf;
	int size;

	f = fopen(fullpath, "rt");

	if (!f) {
		interp->PutStr("Can't open file!");
	    return JS_TRUE;
	}

	fseek (f , 0 , SEEK_END);
    size = ftell (f);
    rewind (f);

	tbuf = new char[size];

	fread (tbuf,1,size,f);

	fclose (f);

	interp->PutStr(tbuf);
	interp->PutStr("\n");

	delete[]tbuf;

	return JS_TRUE;
}

static JSClass global_class = { 
	"global", JSCLASS_HAS_PRIVATE, 
		JS_PropertyStub, JS_PropertyStub,
		global_getProperty, JS_PropertyStub, 
		JS_EnumerateStub, global_resolve,
		JS_ConvertStub, JS_FinalizeStub 
};

static int
DumpAtom(JSHashEntry *he, int i, void *arg)
{
    FILE *fp = (FILE *) arg;
    JSAtom *atom = (JSAtom *)he;

	MScriptInterpreter *interp = (MScriptInterpreter *)arg;

	char sbuf[128];
    sprintf(sbuf, "%3d %08x %5lu ",
            i, (uintN)he->keyHash, (unsigned long)atom->number);
	interp->PutStr(sbuf);
    if (ATOM_IS_STRING(atom)) {
        sprintf(sbuf, "\"%s\"\n", ATOM_BYTES(atom));
		interp->PutStr(sbuf);
    } else if (ATOM_IS_INT(atom)) {
        sprintf(sbuf, "%ld\n", (long)ATOM_TO_INT(atom));
		interp->PutStr(sbuf);
    } else {
        sprintf(sbuf, "%.16g\n", *ATOM_TO_DOUBLE(atom));
		interp->PutStr(sbuf);
	}
    return HT_ENUMERATE_NEXT;
}

static void
DumpScope(JSContext *cx, JSObject *obj, MScriptInterpreter *interp)
{
    uintN i;
    JSScope *scope;
    JSScopeProperty *sprop;
	char sbuf[128];

    i = 0;
    scope = OBJ_SCOPE(obj);
    for (sprop = SCOPE_LAST_PROP(scope); sprop; sprop = sprop->parent) {
        if (SCOPE_HAD_MIDDLE_DELETE(scope) && !SCOPE_HAS_PROPERTY(scope, sprop))
            continue;
        sprintf(sbuf, "%3u %p", i, sprop);
		interp->PutStr(sbuf);
        if (sprop->id & JSVAL_INT) {
            sprintf(sbuf, " [%ld]", (long)JSVAL_TO_INT(sprop->id));
			interp->PutStr(sbuf);
        } else {
            sprintf(sbuf, " \"%s\"", ATOM_BYTES((JSAtom *)sprop->id));
			interp->PutStr(sbuf);
		}

#if 0
#define DUMP_ATTR(name) if (sprop->attrs & JSPROP_##name) fputs(" " #name, fp)
        DUMP_ATTR(ENUMERATE);
        DUMP_ATTR(READONLY);
        DUMP_ATTR(PERMANENT);
        DUMP_ATTR(EXPORTED);
        DUMP_ATTR(GETTER);
        DUMP_ATTR(SETTER);
#undef  DUMP_ATTR
#endif

#if 0
        sprintf(sbuf, " slot %lu flags %x shortid %d\n",
                sprop->slot, sprop->flags, sprop->shortid);
		interp->PutStr(sbuf);
#else
		interp->PutStr("\n");
#endif
    }
}

static JSBool
DumpStats(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
    uintN i;
    JSString *str;
    const char *bytes;
    JSAtom *atom;
    JSObject *obj2;
    JSProperty *prop;
    jsval value;

	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);

    for (i = 0; i < argc; i++) {
        str = JS_ValueToString(cx, argv[i]);
        if (!str)
            return JS_FALSE;
        bytes = JS_GetStringBytes(str);
        
		if (strcmp(bytes, "atom") == 0) {
            interp->PutStr("\natom table contents:\n");
            JS_HashTableDump(cx->runtime->atomState.table, DumpAtom, (FILE *)interp);
        } else if (strcmp(bytes, "global") == 0) {
            DumpScope(cx, cx->globalObject, interp);
        } else {
            atom = js_Atomize(cx, bytes, JS_GetStringLength(str), 0);
            if (!atom)
                return JS_FALSE;
            if (!js_FindProperty(cx, (jsid)atom, &obj, &obj2, &prop))
                return JS_FALSE;
            if (prop) {
                OBJ_DROP_PROPERTY(cx, obj2, prop);
                if (!OBJ_GET_PROPERTY(cx, obj, (jsid)atom, &value))
                    return JS_FALSE;
            }
            if (!prop || !JSVAL_IS_OBJECT(value)) {
				char sbuf[128];
                sprintf(sbuf, "js: invalid stats argument %s\n",
                        bytes);
				interp->PutStr(sbuf);
                continue;
            }
            obj = JSVAL_TO_OBJECT(value);
            if (obj)
                DumpScope(cx, obj, interp);
        }
    }
    return JS_TRUE;
}

static JSFunctionSpec shell_functions[] = {
	{"cat",				Cat,			1},
	{"gc",				GC,				0},
    {"load",            Load,           1},
    {"ls",			    LS,		        0},
    {"cd",			    CD,		        1},
    {"print",           Print,          0},
	{"stats",			DumpStats,		1},
    {0}
};

void MScriptInterpreter::Initialize()
{
	// Define: runtime, context, global object, and standard classes
	rt = JS_NewRuntime(0x100000); 
	cx = JS_NewContext(rt, 0x1000);
	global = JS_NewObject(cx, &global_class, NULL, NULL);
	JS_InitStandardClasses(cx, global);
    JS_SetErrorReporter(cx, aztec_ErrorReporter);
    JS_DefineFunctions(cx, global, shell_functions);
	JS_SetContextPrivate(cx, this);
	JS_SetPrivate(cx, global, this);

	// Add a color class (reflection of MRGBAFloat) to the global namespace
	addRGBAFloatClass(cx);

	// Add a vector class (reflection of MVector3) to the global namespace
	addVector3Class(cx);

	// Add a matrix class (reflection of MMatrix4) to the global namespace
	addMatrix4Class(cx);

	// Add a vector parameter class (reflection of MVector3 parameter)
	addVectorParamClass(cx);

	// Add a mesh vertex class (reflection of MVertex) to the global namespace
	addVertexClass(cx);

	// Add a mesh triangle class (reflection of MTriangle) to the global namespace
	addTriangleClass(cx);

	// Add a mesh class (reflection of MMesh) to the global namespace
	addMeshClass(cx);

	// Add a scene object class (reflection of MNamedObject) to the global namespace
	addSceneObjectClass(cx);

	// Add the scene object to the global namespace
	addSceneClass(cx);

  addArrayParamClass(cx);
}

//--------------------------------------------------------------
// MScriptInterpreter
//--------------------------------------------------------------
MScriptInterpreter::MScriptInterpreter()
{
	m_callback = NULL;
	initFlag = false;
}

MScriptInterpreter::~MScriptInterpreter()
{
	if (initFlag) {

    if (cx)	JS_DestroyContext(cx);
    if (rt)	JS_DestroyRuntime(rt);
    JS_ShutDown();
  }
}

MScriptInterpreter *globalInstance = NULL;

MScriptInterpreter* MScriptInterpreter::getInstance() {
  if (globalInstance == NULL) {
    globalInstance = new MScriptInterpreter;
  }
  return globalInstance;
}

void MScriptInterpreter::cleanInstance() {
  if (globalInstance != NULL) {
    delete globalInstance;
    globalInstance = NULL;
  }
}


JSContext *MScriptInterpreter::GetScriptContext()
{
	if (!initFlag) {
		Initialize();
		initFlag = true;
	}

	return cx;
}

JSObject *MScriptInterpreter::GetGlobalObject()
{
	if (!initFlag) {
		Initialize();
		initFlag = true;
	}

	return global;
}

bool MScriptInterpreter::IsCompilableScript(const char *Script)
{
	if (!initFlag) {
		Initialize();
		initFlag = true;
	}

	return (JS_BufferIsCompilableUnit(cx, global, Script, strlen(Script)) != 0);
}

bool MScriptInterpreter::ExecuteScript(const char *Script, std::string &result)
{
  // always wrap scrpting calls in an undo buffer.
  MSystemManager::getInstance()->getUndoManager()->beginUndo("Action");

  if (!initFlag) {
		Initialize();
		initFlag = true;
	}

	jsval rval;
	JSBool ok = JS_EvaluateScript(cx, global, Script, strlen(Script), NULL, 0, &rval);
	
	if (ok && (rval != JSVAL_VOID)) {
		/* Suppress error reports from JS_ValueToString(). */
		JSErrorReporter older = JS_SetErrorReporter(cx, NULL);
		JSString *resStr = JS_ValueToString(cx, rval);
		JS_SetErrorReporter(cx, older);

		if (resStr != NULL) {
			result = JS_GetStringBytes(resStr);
		} else {
			result = "";
			ok = false;
		}
	} else {
		// Script failed
		result = "";
	}
  Redraw();

  // Finish off our lot of undo.
  MSystemManager::getInstance()->getUndoManager()->endUndo();

	return ((ok != 0) ? true : false);
}

bool MScriptInterpreter::ExecuteScript(const char *Script) {
  std::string result;
  return ExecuteScript(Script, result);
}

void MScriptInterpreter::putCallbacks(CScriptCallbacks *e)
{
	m_callback = e;
}

int MScriptInterpreter::PutStr(const char *str)
{
	if (m_callback != NULL) {
		return m_callback->puts(str);
	} else {
		return 0;
	}
}

void MScriptInterpreter::Redraw()
{
	if (m_callback != NULL) {
		m_callback->updateDisplay();
	}
}

CScriptCallbacks::actionFunc *
MScriptInterpreter::GetAction(const char *str)
{
	if (m_callback != NULL) {
		return m_callback->getActionFunction(str);
	} else {
		return NULL;
	}
}

CScriptCallbacks::actionFuncWithArgs *
MScriptInterpreter::getActionWithArgs(const char *str)
{
	if (m_callback != NULL) {
		return m_callback->getActionFunctionWithArgs(str);
	} else {
		return NULL;
	}
}


	// After loading plugin DLLs, call this to see if any of them extend scripting.
	void extendScripting(MScriptInterpreter *interp);


static void* dllGetFunc(HINSTANCE lib, const char *funcName) {
#ifdef WIN32
	return (void*)GetProcAddress(lib, funcName);
#else
	return (void*)dlsym(lib, funcName);
#endif
}


void MScriptInterpreter::extendScripting(MSystemManager *sysMan)
{
	MBaseObjectListPtr dllList = sysMan->getDllList();
	dllList->beginIteration();
	MBaseObjectPtr dllObj;
	
	while ((dllObj = dllList->getNext()) != NULL) {
		MDllFilePtr dll = AZTEC_CAST(MDllFile, dllObj);
		
		if (dll == NULL) {
			continue;
		}
		
		// Assuming it was an otherwise valid plugin, lets see if it wants to implement
		// some scripting.
		ScriptingFunc ScriptFunc = (ScriptingFunc)dllGetFunc(dll->getInstance(), "ModScripting");
		if (ScriptFunc) {
			// Yes - the plugin wants to add something to the scripting engine.
			ScriptFunc(GetScriptContext(), GetGlobalObject());
		}
	}
	dllList->endIteration();
}

  
}
