#include "StdAfx.h"

#include <mesh/MVertexArrayParam.h>
#include <mesh/MTriangleArrayParam.h>
#include <mesh/ComponentFlagParam.h>
#include <MStr.h>

#include <assert.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

std::vector<MStr> MeshTriangleElement::memberNames;

MeshTriangleElement::MeshTriangleElement(MMesh *mesh, int index) {
	// if we haven't initialised out member names before, do it now
	if (memberNames.size() == 0) {
		memberNames.push_back("indexA");
		memberNames.push_back("indexB");
		memberNames.push_back("indexC");
		memberNames.push_back("vertexA");
		memberNames.push_back("vertexB");
		memberNames.push_back("vertexC");
		memberNames.push_back("edgeflagsAB");
		memberNames.push_back("edgeflagsBC");
		memberNames.push_back("edgeflagsCA");
	}

	this->mesh = mesh;
	this->triangleIndex = index;
	this->editMeshCheckDone = false;
	this->editMesh = NULL;

	setStorable(true);
}

// MAggregateParameter methods
int MeshTriangleElement::getMemberCount() {
	return memberNames.size();
}

MParameterObjectPtr MeshTriangleElement::getMember(int index) {
	if (index >= 0 && index < 3) {
		return new MeshTriangleIndex(mesh, triangleIndex, index);            
	} else if (index >= 3 && index < 6) {
		return new MeshVertexPosParam(mesh, mesh->getTriVert(triangleIndex, index-3), false);
  } else if (index == 6) {
    return new ComponentFlagParam("edgeflagsAB", mesh, MComponentisedObject::EDGE_TYPE, triangleIndex * 3);
  } else if (index == 7) {
    return new ComponentFlagParam("edgeflagsBC", mesh, MComponentisedObject::EDGE_TYPE, triangleIndex * 3 + 1);
  } else if (index == 8) {
    return new ComponentFlagParam("edgeflagsCA", mesh, MComponentisedObject::EDGE_TYPE, triangleIndex * 3 + 2);
  }

	return NULL;
}

MStr MeshTriangleElement::getMemberName(int index) {
	return memberNames[index];
}

int MeshTriangleElement::getMemberIndex(const MStr &name) {
	for (unsigned int i = 0; i < memberNames.size(); ++i) {
		if (memberNames[i].compareNoCase(name) == 0) {
			return i;
		}
	}
	return -1;
}

// MParameterObject methods
int MeshTriangleElement::getDataType() const {
	return TYPE_AGGREGATE;
}

int MeshTriangleElement::getDataMeaning() const {
	return MEANING_ANY;
}

MStr MeshTriangleElement::getShortName() const {
	char buf[64];
	sprintf(buf, "tris[%i]", triangleIndex);
	return buf;
}

MStr MeshTriangleElement::getLongName() const {
	char buf[64];
	sprintf(buf, "tris[%i]", triangleIndex);
	return buf;
}

MStr MeshTriangleElement::getFriendlyName() const {
	char buf[64];
	sprintf(buf, "tris[%i]", triangleIndex);
	return buf;
}

MBaseObject* MeshTriangleElement::getOwner() const {
	return mesh;
}

void MeshTriangleElement::setOwner(MBaseObject *newOwner) {
	assert(0);
}

bool MeshTriangleElement::isInputParameter() {
	return true;
}

bool MeshTriangleElement::getValueString(MStr &value) {
	if (triangleIndex < 0 || triangleIndex >= mesh->getNumVerts()) {
		return false;
	}

	value = mesh->getVertexPosition(triangleIndex).convertToString();
	return true;
}

bool MeshTriangleElement::getValueInteger(int &value) { 
	return false; 
}

bool MeshTriangleElement::getValueFloat(float &value) { 
	return false; 
}

bool MeshTriangleElement::getValueBoolean(bool &value) { 
	return false; 
}

bool MeshTriangleElement::getValueVector(MVector3 &value) {
	return false;
}

bool MeshTriangleElement::setValueString(const MStr &value) { return false; }
bool MeshTriangleElement::setValueInteger(int value) { return false; }
bool MeshTriangleElement::setValueFloat(float value) { return false; }
bool MeshTriangleElement::setValueBoolean(bool value) { return false; }

bool MeshTriangleElement::setValueVector(const MVector3 &value) { 
	return false;
}

bool MeshTriangleElement::createKeyAt(long time) { 
	return false; 
}

bool MeshTriangleElement::setValueParameter(const MParameterObjectPtr &src) {
	return false;
}

bool MeshTriangleElement::setFromParameter(MParameterObjectPtr src) {
	return setValueParameter(src);
}

bool MeshTriangleElement::isConnectable() { 
	return false; 
}
bool MeshTriangleElement::isConnected() { 
	return false; 
}

bool MeshTriangleElement::setInputParameter(const MParameterObjectPtr &input)  { 
	return false; 
}

MParameterObjectPtr MeshTriangleElement::getInputParameter() { 
	return NULL; 
}

MParameterObjectPtr MeshTriangleElement::createCopy() { 
	return MParameterFactory::createVector3(getShortName(), getLongName(), getFriendlyName());
}

bool MeshTriangleElement::isEditableMesh() {
	if (!editMeshCheckDone) {
		editMesh = AZTEC_CAST(MEditableMesh, mesh);
		editMeshCheckDone = true;
	}

	return editMesh != NULL;
}

MTriangleArrayParam::MTriangleArrayParam( MMesh *mesh ) 
	: MParameterObjectImpl("tris", "triangles", "Triangles", NULL),
	mesh(mesh),
	editMesh(NULL),
	editMeshCheckDone(false)
{
}

MTriangleArrayParam::~MTriangleArrayParam() {
}

MBaseObject* MTriangleArrayParam::getOwner() const {
	return mesh;
}

void MTriangleArrayParam::setOwner(MBaseObject *newOwner) {
	assert(newOwner == mesh);
}

bool MTriangleArrayParam::isInputParameter() {
	return true;
}

int MTriangleArrayParam::getElementCount() {
	return mesh->getNumTris();
}

int MTriangleArrayParam::addElement() {
	if (isEditableMesh()) {
		editMesh->addTriangle(-1,-1,-1);
		return editMesh->getNumTris() - 1;
	}
	return -1;
}

void MTriangleArrayParam::removeElement(int index) {
	if (isEditableMesh()) {
		editMesh->deleteTriangle(index);
	}
}

void MTriangleArrayParam::resize(int size) {
	if (isEditableMesh()) {
		while (editMesh->getNumTris() > size) {
			editMesh->deleteTriangle(editMesh->getNumTris()-1);
		}

		// add any vertices that we need to.
		while (editMesh->getNumTris() < size) {
			editMesh->addTriangle(-1,-1,-1);
		}
	}
}

MParameterObjectPtr MTriangleArrayParam::getElement(int index) {
	return new MeshTriangleElement(mesh, index);
}

int MTriangleArrayParam::getElementType() const { 
	return TYPE_AGGREGATE; 
}

// MParameterObject methodsfs
int MTriangleArrayParam::getDataMeaning() const { 
	return MEANING_ANY; 
}

bool MTriangleArrayParam::createKeyAt(long time) { 
	return false;
}

MParameterObjectPtr MTriangleArrayParam::createCopy() {
	// TODO: return something that can contain a copy
	return NULL;
}

bool MTriangleArrayParam::setValueParameter(const MParameterObjectPtr &src) {
  // this actually has no meaning, so we sort of leave this out for now.
  return false;
}


bool MTriangleArrayParam::isEditableMesh() {
	if (!editMeshCheckDone) {
		editMesh = AZTEC_CAST(MEditableMesh, mesh);
		editMeshCheckDone = true;
	}

	return editMesh != NULL;
}


MeshTriangleIndex::MeshTriangleIndex(MMesh *mesh, int triIndex, int vertIndex) 
{
	this->mesh = mesh;
	this->triangleIndex = triIndex;
	this->vertexIndex = vertIndex;
	this->editMesh = NULL;
	this->editMeshCheckDone = false;
	setStorable(true);
}

// MIntParameter methods
int MeshTriangleIndex::getValue() {
	return mesh->getTriVert(triangleIndex, vertexIndex);
}

// MParameterObject methods
int MeshTriangleIndex::getDataType() const {
	return TYPE_INTEGER;
}

int MeshTriangleIndex::getDataMeaning() const {
	return MEANING_ANY;
}

MStr MeshTriangleIndex::getShortName() const {
	switch (vertexIndex) {
case 0: return "indexA";
case 1: return "indexB";
case 2: return "indexC";
	}
	return "";
}

MStr MeshTriangleIndex::getLongName() const {
	return getShortName();
}

MStr MeshTriangleIndex::getFriendlyName() const {
	return getShortName();
}

MBaseObject* MeshTriangleIndex::getOwner() const {
	return mesh;
}

void MeshTriangleIndex::setOwner(MBaseObject *newOwner) {
	assert(newOwner == mesh);
}

bool MeshTriangleIndex::isInputParameter() {
	return true;
}

bool MeshTriangleIndex::getValueString(MStr &value) {
	value = intToStr(getValue());
	return true;
}

bool MeshTriangleIndex::getValueInteger(int &value) {
	value = getValue();
	return true;
}

bool MeshTriangleIndex::getValueFloat(float &value) {
	value = (float)getValue();
	return true;
}

bool MeshTriangleIndex::getValueBoolean(bool &value) {
	return false;
}

bool MeshTriangleIndex::getValueVector(MVector3 &value) {
	return false;
}

bool MeshTriangleIndex::setValueString(const MStr &value) {
	return setValueInteger(strToInt(value));
}

bool MeshTriangleIndex::setValueInteger(int value) {
	if (!isEditableMesh()) {
		return false;
	}

	editMesh->setTriangleVertex(triangleIndex, vertexIndex, value);
	mesh->flagOutputs(OBJECTFLAG_NEEDS_UPDATE);

	return true;
}

bool MeshTriangleIndex::setValueFloat(float value) {
	return setValueInteger((int)value);
}

bool MeshTriangleIndex::setValueBoolean(bool value) {
	return false;
}

bool MeshTriangleIndex::setValueVector(const MVector3 &value) {
	return false;
}

bool MeshTriangleIndex::setValueParameter(const MParameterObjectPtr &src) {
	int value;
	if (src->getValueInteger(value)) {
		return setValueInteger(value);
	}

	return false;
}

bool MeshTriangleIndex::createKeyAt(long time) {
	// Mesh triangle indeicies are not animatable.
	return false;
}

bool MeshTriangleIndex::setFromParameter(MParameterObjectPtr src) {
	int value;
	if (src->getValueInteger(value)) {
		return setValueInteger(value);
	}

	return false;
}

bool MeshTriangleIndex::isConnectable() {
	return false;
}

bool MeshTriangleIndex::isConnected() {
	return false;
}

bool MeshTriangleIndex::setInputParameter(const MParameterObjectPtr &input) {
	return false;
}

MParameterObjectPtr MeshTriangleIndex::getInputParameter() {
	return NULL;
}

MParameterObjectPtr MeshTriangleIndex::createCopy() {
	return MParameterFactory::createInteger(getShortName(), getLongName(), getFriendlyName());
}

bool MeshTriangleIndex::isEditableMesh() {
	if (!editMeshCheckDone) {
		editMesh = AZTEC_CAST(MEditableMesh, mesh);
		editMeshCheckDone = true;
	}

	return editMesh != NULL;
}



}


