#include "StdAfx.h"

#include "MListsTrees.h"
#include "MNamedObject.h"

#include <assert.h>

#define MSORTEDLIST_GRAN      16

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


namespace Aztec {
#define CHECK      assert((m_Head == NULL && m_Tail == NULL) || (m_Head != NULL && m_Tail != NULL));

  
  //----------------------------------------------------------------------------------------
  //  MListObjectNode
  //----------------------------------------------------------------------------------------
  
  MListObjectNode::MListObjectNode() {
    m_Next = NULL;
    m_Prev = NULL;
  }
  
  MListObjectNode::~MListObjectNode() {
  }

  void MListObjectNode::finishWithUndoNode() {
    undo = NULL;
  }

  MListObjectNode::Undo::Undo(MListObjectNode *obj) {
    prev = obj->m_Prev;
    next = obj->m_Next;
    object = obj;
  }


  MUndoableObject* MListObjectNode::Undo::getObject() {
    return &*object;
  }

  void MListObjectNode::Undo::undo() {
    std::swap(object->m_Prev, prev);
    std::swap(object->m_Next, next);
  }

  void MListObjectNode::Undo::redo() {
    std::swap(object->m_Prev, prev);
    std::swap(object->m_Next, next);
  }

  
  //----------------------------------------------------------------------------------------
  //  MTreeObjectNode
  //----------------------------------------------------------------------------------------
  
  MTreeObjectNode::MTreeObjectNode() {
    m_Parent = NULL;
    m_FirstChild = NULL;
    m_NextSibling = NULL;
    m_PrevSibling = NULL;
  }
  
  MTreeObjectNode::~MTreeObjectNode() {
  }
  
  int MTreeObjectNode::isChildOf(const MTreeObjectNodePtr &Node) {
    MTreeObjectNodePtr TempNode;
    int               Count;
    
    Count = 0;
    TempNode = m_Parent;
    
    while (TempNode != NULL) {
      Count++;
      
      if (Node == TempNode) {
        return Count;
      }
      TempNode = TempNode->getParent();
    }
    
    if (Node == TempNode) {
      return Count;
    }
    
    return 0;
  }
  
  int MTreeObjectNode::isChildOfObject(const MBaseObjectPtr &Obj) {
    MTreeObjectNodePtr TempNode;
    int               Count;
    
    Count = 0;
    TempNode = this;
    
    while (TempNode != NULL) {
      Count++;
      
      if (TempNode->getObject() == Obj) {
        return Count;
      }
      TempNode = TempNode->getParent();
    }
    
    return 0;
  }
  
  int MTreeObjectNode::isChildOfFlagged(AztecFlags Flags)
  {
    MTreeObjectNodePtr Node;
    MBaseObjectPtr Obj;
    
    Node = this;
    
    while (Node != NULL) {
      Obj = Node->getObject();
      
      if (Obj == NULL) {
        Node = Node->getParent();
      }
      
      if (Obj->isFlagged(Flags)) {
        return 1;
      }
      
      Node = Node->getParent();
    }
    
    return 0;
  }
  
  void MTreeObjectNode::finishWithUndoNode() {
    undo = NULL;
  }

  MTreeObjectNode::Undo::Undo(MTreeObjectNode *obj) {
    parent = obj->m_Parent;
    child = obj->m_FirstChild;
    prev = obj->m_PrevSibling;
    next = obj->m_NextSibling;
    object = obj;
  }


  MUndoableObject* MTreeObjectNode::Undo::getObject() {
    return &*object;
  }

  void MTreeObjectNode::Undo::undo() {
    std::swap(object->m_FirstChild, child);
    std::swap(object->m_Parent, parent);
    std::swap(object->m_PrevSibling, prev);
    std::swap(object->m_NextSibling, next);
  }

  void MTreeObjectNode::Undo::redo() {
    std::swap(object->m_FirstChild, child);
    std::swap(object->m_Parent, parent);
    std::swap(object->m_PrevSibling, prev);
    std::swap(object->m_NextSibling, next);
  }

  
  ListData::ListData() {
    m_Count = 0;
  }

  ListData::ListData(const ListData &src) {
    assert((src.m_Head == NULL && src.m_Tail == NULL) || (src.m_Head != NULL && src.m_Tail != NULL));
    
    m_Head = src.m_Head;
    m_Tail = src.m_Tail;
    m_CurPos = src.m_CurPos;
    m_Count = src.m_Count;
    m_PosStack = src.m_PosStack;

    assert((m_Head == NULL && m_Tail == NULL) || (m_Head != NULL && m_Tail != NULL));
  }

  ListData::~ListData() {
  }
  
  ListData& ListData::operator=(const ListData &src) {
    assert((src.m_Head == NULL && src.m_Tail == NULL) || (src.m_Head != NULL && src.m_Tail != NULL));

    m_Head = src.m_Head;
    m_Tail = src.m_Tail;
    m_CurPos = src.m_CurPos;
    m_Count = src.m_Count;
    m_PosStack = src.m_PosStack;

    assert((m_Head == NULL && m_Tail == NULL) || (m_Head != NULL && m_Tail != NULL));
    return *this;
  }

  //----------------------------------------------------------------------------------------
  //  MBaseObjectList
  //----------------------------------------------------------------------------------------
  
  MBaseObjectList::MBaseObjectList() {
    m_Head = NULL;
    m_Tail = NULL;
    m_Count = 0;
    m_CurPos = NULL;
    undoEnabled = true;
  }
  
  MBaseObjectList::~MBaseObjectList() {
    undoEnabled = false;
    removeAllNodes();
  }
  
  MBaseObjectPtr MBaseObjectList::createNew() {
    MBaseObjectListPtr  NewList;
    MListObjectNodePtr  SrcNode, LastNewNode, NewNode;
    
    NewList = new MBaseObjectList;
    
    SrcNode = m_Head;
    NewNode = NULL;
    LastNewNode = NULL;
    
    // Repeat for every node in the source list.
    while (SrcNode != NULL) {
      NewNode = new MListObjectNode;
      NewNode->setObject(SrcNode->getObject());
//      NewNode->setFlag(SrcNode->getFlags());
      
      if (SrcNode->getPrev() == NULL)     // we are on the first node
        NewList->m_Head = NewNode;
      
      NewNode->setPrev(LastNewNode);
      if (LastNewNode != NULL)
        LastNewNode->setNext(NewNode);
      
      LastNewNode = NewNode;
      
      m_Count++;
      SrcNode = SrcNode->getNext();
    }
    
    NewList->m_Tail = NewNode;
    
    return NewList;
  }
  
  MListObjectNodePtr MBaseObjectList::addHead(MBaseObjectPtr Obj) {
    CHECK;

    if (Obj == NULL) {
      return NULL;
    }

    ensureUndo();
    
    MListObjectNodePtr Node;
    
    Node = new MListObjectNode;
    Node->setObject(Obj);
    
    Node->setPrev(NULL);
    Node->setNext(m_Head);
    
    if (m_Head != NULL) {
      m_Head->setPrev(Node);
      m_Head = Node;
    } else {
      m_Head = Node;
      m_Tail = Node;
    }
    
    m_Count++;
    
    CHECK;

    return Node;
  }
  
  MListObjectNodePtr MBaseObjectList::addTail(MBaseObjectPtr Obj) {
    if (Obj == NULL) {
      return NULL;
    }
    
    CHECK;

    ensureUndo();

    MListObjectNodePtr Node;
    
    Node = new MListObjectNode;
    Node->setObject(Obj);
    
    
    Node->setPrev(m_Tail);
    Node->setNext(NULL);
    
    if (m_Tail != NULL) {
      m_Tail->setNext(Node);
      m_Tail = Node;
    } else {
      m_Head = Node;
      m_Tail = Node;
    }
    
    m_Count++;
    
    CHECK;

    return Node;
  }
  
  MListObjectNodePtr MBaseObjectList::insertAfter(MListObjectNodePtr After, MBaseObjectPtr Obj) {
    if (After == NULL) {
      addHead(Obj);
      return NULL;
    }
    
    CHECK;
    ensureUndo();

    MListObjectNodePtr Node;
    
    Node = new MListObjectNode;
    Node->setObject(Obj);
    
    Node->setPrev(After);
    
    if (After == m_Tail) {  // are we inserting after the tail node
      m_Tail = Node;
    }
    
    Node->setNext(After->getNext());
    After->setNext(Node);
    CHECK;
    
    return Node;
  }
  
  MListObjectNodePtr MBaseObjectList::findNode(MBaseObjectPtr Obj) {
    MListObjectNodePtr Node;
    Node = m_Head;
    
    while (Node != NULL) {
      if (Node->getObject() == Obj) {
        return Node;
      }
      
      Node = Node->getNext();
    }
    
    // No matching node found, return NULL
    return NULL;
  }
  
  void MBaseObjectList::removeNode(MListObjectNodePtr Node) {
    // join the two surrounding nodes together
    if (Node == NULL)
      return;
    
    ensureUndo();

    MListObjectNodePtr next = Node->getNext();
    MListObjectNodePtr prev = Node->getPrev();

    if (Node->getNext() != NULL)
      Node->getNext()->setPrev(prev);
    if (Node->getPrev() != NULL)
      Node->getPrev()->setNext(next);
    
    if (Node == m_Head)
      m_Head = next;
    
    if (Node == m_Tail)
      m_Tail = prev;
    
    Node = NULL;
    
    restartIteration();
    
    m_Count--;
  }
  
  void MBaseObjectList::removeAllNodes() {
    CHECK;
    ensureUndo();

    MListObjectNodePtr Node, NextNode;
    
    Node = m_Head;
    
    while (Node != NULL) {
      NextNode = Node->getNext();

      // Don't leave dangling references - they keep smart pointers from
      // knowing when to clean up
      Node->setNext(NULL);

      Node = NextNode;
    }
    
    m_Head = NULL;
    m_Tail = NULL;
    m_Count = 0;

    CHECK;
  }
  
  
  MBaseObjectPtr MBaseObjectList::getHead() {
    if (m_Head == NULL)
      return NULL;
    
    return m_Head->getObject();
  }
  
  MBaseObjectPtr MBaseObjectList::getTail() {
    if (m_Tail == NULL)
      return NULL;
    
    return m_Tail->getObject();
  }
  
  MListObjectNodePtr MBaseObjectList::getHeadNode() {
    return m_Head;
  }
  
  MListObjectNodePtr MBaseObjectList::getTailNode() {
    return m_Tail;
  }
  
  
  MBaseObjectPtr MBaseObjectList::removeHead() {
    if (m_Head == NULL) {
      return NULL;
    }

    CHECK;
    ensureUndo();

    MBaseObjectPtr      Obj;
    MListObjectNodePtr  Node;
    
    Node = m_Head;
    
    if (m_Head->getNext() != NULL) {    // there is more than one node in the list
      m_Head = m_Head->getNext();
      m_Head->setPrev(NULL);

    } else {    // there is only one node
      m_Head = NULL;
      m_Tail = NULL;
    }
    
    Obj = Node->getObject();
    
    if (Obj->getRefCount() < 1) {
      Obj = NULL;
    }

    Node = NULL;
    m_Count--;

    CHECK;
    return Obj;
  }
  
  MBaseObjectPtr MBaseObjectList::removeTail()
  {
    if (m_Head == NULL) {
      return NULL;
    }
    
    CHECK;
    ensureUndo();

    MBaseObjectPtr Obj;
    MListObjectNodePtr Node;
    
    Node = m_Tail;
    
    if (m_Tail->getPrev() != NULL) {   // there is more than one node in the list
      m_Tail = m_Tail->getPrev();
      m_Tail->setNext(NULL);

    } else {    // there is only one node
      m_Head = NULL;
      m_Tail = NULL;
    }
    
    Obj = Node->getObject();
    
    if (Obj->getRefCount() < 1) {
      Obj = NULL;
    }
    
    Node = NULL;
    m_Count--;
    
    CHECK;
    return Obj;
  }
  
  int MBaseObjectList::getCount() {
    CHECK;

    int               Num;
    MListObjectNodePtr Node;
    
    Num = 0;
    Node = m_Head;
    while (Node != NULL) {
      Num++;
      Node = Node->getNext();
    }
    
    return Num;
    //   return m_Count;
  }
  
  bool MBaseObjectList::isEmpty() {
    return (m_Count == 0);
  }
  
  void MBaseObjectList::beginIteration() {
    m_PosStack.push(m_CurPos);
    m_CurPos = NULL;
  }

  void MBaseObjectList::endIteration() {
    m_CurPos = m_PosStack.top();
    m_PosStack.pop();
  }

  void MBaseObjectList::restartIteration() {
    m_CurPos = NULL;
  }
  
  MBaseObjectPtr MBaseObjectList::getNext() {
    if (m_Head == NULL) {
      return NULL;
    }
    
    if (m_CurPos == NULL) {
      m_CurPos = m_Head;
      return m_CurPos->getObject();
    }
    
    m_CurPos = m_CurPos->getNext();
    if (m_CurPos != NULL) {
      return m_CurPos->getObject();
    }

    return NULL;
  }
  
  MListObjectNodePtr MBaseObjectList::findObjectWithClass(MStr Name) {
    MListObjectNodePtr Node;
    
    Node = m_Head;
    while (Node != NULL) {
      if (Node->getObject()->getClassName() == Name) {
        return Node;
      }
      
      Node = Node->getNext();
    }
    return NULL;
  }
  
  MObjectNodePtr MBaseObjectList::enumList(MEnumFunc Func, AztecFlags lParam) {
    MListObjectNodePtr Node;

    Node = m_Head;
    
    while (Node != NULL) {
      if (Func(Node, lParam) == 0) {
        return Node;
      }
      Node = Node->getNext();
    }
    
    return NULL;
  }
  
  void MBaseObjectList::finishWithUndoNode() {
    nodeUndo = NULL;
    MBaseObject::finishWithUndoNode();
  }               

  void MBaseObjectList::ensureUndo() {
    if (undoEnabled) {
      doEnsureUndo(nodeUndo, this);
    }
  }

  MBaseObjectList::Undo::Undo(MBaseObjectList *node) {
    object = node;
    stored.m_Count = node->m_Count;
    stored.m_Head = node->m_Head;
    stored.m_Tail = node->m_Tail;
    stored.m_CurPos = node->m_CurPos;
    stored.m_PosStack = node->m_PosStack;
  }

  MUndoableObject* MBaseObjectList::Undo::getObject() {
    return &*object;
  }

  void MBaseObjectList::Undo::undo() {
    std::swap(stored.m_Count, object->m_Count);
    std::swap(stored.m_Head, object->m_Head);
    std::swap(stored.m_Tail, object->m_Tail);
    std::swap(stored.m_CurPos, object->m_CurPos);
    std::swap(stored.m_PosStack, object->m_PosStack);
  }

  void MBaseObjectList::Undo::redo() {
    std::swap(stored.m_Count, object->m_Count);
    std::swap(stored.m_Head, object->m_Head);
    std::swap(stored.m_Tail, object->m_Tail);
    std::swap(stored.m_CurPos, object->m_CurPos);
    std::swap(stored.m_PosStack, object->m_PosStack);
  }



  TreeData::TreeData() {
    m_Count = 0;
    m_Root = NULL;
  }

  TreeData::TreeData(const TreeData &src) {
    m_Count = src.m_Count;
    m_Root = src.m_Root;
  }

  TreeData::~TreeData() {
  }
  
  TreeData& TreeData::operator=(const TreeData &rhs) {
    m_Count = rhs.m_Count;
    m_Root = rhs.m_Root;
    return *this;
  }
  
  //----------------------------------------------------------------------------------------
  //  MBaseObjectTree
  //----------------------------------------------------------------------------------------
  
  MBaseObjectTree::MBaseObjectTree() {
    m_Root = NULL;
    m_Count = 0;
    
    m_CurIter.m_LastNode = NULL;
    undoEnabled = true;
  }
  
  MBaseObjectTree::~MBaseObjectTree() {
    undoEnabled = false;
    MTreeObjectNodePtr Node, Next;
    
    Node = m_Root;
    
    while (Node != NULL) {
      Next = Node->getNextSibling();
      deleteNodeAndChildren(Node);
      Node = Next;
    }
    
    m_CurIter.m_TraverseStack.clear();
  }
  
  MBaseObjectPtr MBaseObjectTree::createNew() {
    MBaseObjectTreePtr Tree;
    MBaseObjectPtr SrcObj, SrcParentObj;
    MTreeObjectNodePtr  SrcObjNode, SrcParentNode, DestParentNode, NewNode;
    
    Tree = new MBaseObjectTree;
    
    // Are there any nodes at all?
    if (m_Root != NULL) {
      beginIteration();
      
      while ((SrcObj = getNext()) != NULL) {
        NewNode = NULL;
        SrcObjNode = getCurrentNode();
        SrcParentNode = SrcObjNode->getParent();
        
        // Are we on a root level node?
        if (SrcParentNode == NULL) {
          NewNode = Tree->addNode(SrcObj, NULL);
        } else {
          SrcParentObj = SrcParentNode->getObject();
          DestParentNode = Tree->findObject(SrcParentObj);
          NewNode = Tree->addNode(SrcObj, DestParentNode);
        }
        if (NewNode != NULL) {
//          NewNode->setFlag(SrcObjNode->getFlags());
        }
      }

      endIteration();
    }

    
    return Tree;
  }
  
  MTreeObjectNodePtr MBaseObjectTree::addNode(MBaseObjectPtr Obj, MTreeObjectNodePtr Parent) {
    ensureUndo();

    MTreeObjectNodePtr Node;
    
    Node = new MTreeObjectNode;
    Node->setObject(Obj);
    
    // TODO: At the momoent, this adds nodes a the end of a list of
    // nodes. To do this, it iterates over the entire list, and places it
    // at the end. We need to store a lastNode member in the each node.

    if (Parent == NULL)      // Insert at the root
    {
      // Place the new node at the end of the root siblings.
     
      if (m_Root != NULL) {
        MTreeObjectNodePtr lastNode = m_Root;
        while (lastNode->getNextSibling() != NULL) {
          lastNode = lastNode->getNextSibling();
        }
        lastNode->setNextSibling(Node);
        Node->setPrevSibling(lastNode);
      } else {
        m_Root = Node;
      }
      
      m_Count ++;
      
      return Node;
    }
    
    if (Parent->getFirstChild() == NULL)      // are we the first child of the Parent node?
    {
      Parent->setFirstChild(Node);
      Node->setParent(Parent);
      m_Count ++;
      
      return Node;
    }

    MTreeObjectNodePtr lastNode = Parent->getFirstChild();
    while (lastNode->getNextSibling() != NULL) {
      lastNode = lastNode->getNextSibling();
    }
    lastNode->setNextSibling(Node);
    Node->setPrevSibling(lastNode);
    Node->setParent(Parent);
    
    m_Count++;
    
    return Node;
  }
  
  void MBaseObjectTree::deleteNodeKeepChildren(MTreeObjectNodePtr Node) {
    if (Node == NULL)
      return;
    
    ensureUndo();

    // Set the parent of each of the children nodes.
    {
      MTreeObjectNodePtr ChildNode, NextChildNode;
      
      ChildNode = Node->getFirstChild();
      while (ChildNode != NULL)
      {
        NextChildNode = ChildNode->getNextSibling();
        setParent(ChildNode, Node->getParent());
        ChildNode = NextChildNode;
      }
    }
    
    // Now that all the children have been moved. We can remove the required node out
    // of the sibling list, and then delete it.
    
    // Move the required node to the Root level and then delete it.
    setParent(Node, NULL);
    
    if (Node->getPrevSibling() != NULL) {
      Node->getPrevSibling()->setNextSibling(Node->getNextSibling());
    }
    
    if (Node->getNextSibling() != NULL) {
      Node->getNextSibling()->setPrevSibling(Node->getPrevSibling());
    }
    
    if (m_Root == Node) {
      m_Root = Node->getNextSibling();
    }
    
    m_Count--;

    Node = NULL;
  }
  
  void MBaseObjectTree::deleteNodeAndChildren(MTreeObjectNodePtr Node) {
    // Use recursion do delete the nodes for now. Perhaps change to stakcs and whatnot
    // later on if function stack size becomes a problem.
    
    if (Node == NULL)
      return;
    
    ensureUndo();

    MTreeObjectNodePtr TempNode, NextNode;
    
    TempNode = Node->getFirstChild();
    while (TempNode != NULL) {
      NextNode = TempNode->getNextSibling();
      deleteNodeAndChildren(TempNode);
      TempNode = NextNode;
    }
    
    if (Node->getNextSibling() != NULL) {
      Node->getNextSibling()->setPrevSibling(Node->getPrevSibling());
    }

    if (Node->getPrevSibling() != NULL) {
      Node->getPrevSibling()->setNextSibling(Node->getNextSibling());
    }
    
    if (Node->getParent() != NULL) {
      if (Node->getParent()->getFirstChild() == Node)
        Node->getParent()->setFirstChild(Node->getNextSibling());
    }
    else
    {
      if (m_Root == Node)
        m_Root = Node->getNextSibling();
    }
    
    m_Count--;
  }
  
  
  void MBaseObjectTree::beginIteration(MTreeTraverseType Type) {
    m_IterStack.push(m_CurIter);

    m_CurIter.m_TraverseType = Type;
    restartIteration();
  }

  void MBaseObjectTree::endIteration() {
    m_CurIter = m_IterStack.top();
    m_IterStack.pop();
  }

  void MBaseObjectTree::restartIteration() {
    m_CurIter.m_LastNode = NULL;
    m_CurIter.m_TraverseStack.clear();
  }
  

  MTreeObjectNodePtr MBaseObjectTree::findObject(MBaseObjectPtr Obj, MTreeObjectNodePtr Node) {
    MTreeObjectNodePtr NewNode, FoundNode;
    
    // Are we looking from the root?
    if (Node == NULL) {
      NewNode = m_Root;
      while (NewNode != NULL) {
        FoundNode = findObject(Obj, NewNode);
        if (FoundNode != NULL) {
          return FoundNode;
        }
        
        NewNode = NewNode->getNextSibling();
      }
      return NULL;
    }
    
    if (Node->getObject() == Obj) {
      return Node;
    }
    
    NewNode = Node->getFirstChild();
    while (NewNode != NULL) {
      while (NewNode != NULL) {
        FoundNode = findObject(Obj, NewNode);
        if (FoundNode != NULL) {
          return FoundNode;
        }
        
        NewNode = NewNode->getNextSibling();
      }
      return NULL;
    }
    return NULL;
  }
  
  MBaseObjectPtr MBaseObjectTree::findObject(const MStr &Name) {
    if (Name.GetLength() == 0) {
      return NULL;
    }
    MBaseObjectPtr Obj;
    
    beginIteration();
    while ((Obj = getNext()) != NULL) {
      MNamedObjectPtr namedObj = AZTEC_CAST(MNamedObject, Obj);
      if (namedObj != NULL) {
        if (namedObj->getName().compareNoCase(Name) == 0) {
          return Obj;
        }
      }
    }
    endIteration();
    
    return NULL;
  }
  
  MTreeObjectNodePtr MBaseObjectTree::findObjectNode(const MStr &Name) {
    MBaseObjectPtr Obj;
    
    beginIteration();
    while ((Obj = getNext()) != NULL) {
      MNamedObjectPtr namedObj = AZTEC_CAST(MNamedObject, Obj);
      if (namedObj != NULL) {
        if (namedObj->getName().compareNoCase(Name) == 0) {
          return getCurrentNode();
        }
      }
    }
    endIteration();
    
    return NULL;
  }
  
  
  MBaseObjectPtr MBaseObjectTree::getNext() {
    if (m_Root == NULL) {
      return NULL;
    }
    
    if (m_CurIter.m_TraverseType == ttParentChildSib) {
      if (m_CurIter.m_LastNode == NULL) {
        m_CurIter.m_LastNode = m_Root;
        return m_CurIter.m_LastNode->getObject();
      }
      
      // Goto the children first
      if (m_CurIter.m_LastNode->getFirstChild() != NULL) {
        if (m_CurIter.m_LastNode->getNextSibling() != NULL) {
          m_CurIter.m_TraverseStack.push_back( m_CurIter.m_LastNode->getNextSibling() );
        }
        m_CurIter.m_LastNode = m_CurIter.m_LastNode->getFirstChild();
        return m_CurIter.m_LastNode->getObject();
      }
      
      // If there is no children, goto the next sibling;
      
      if (m_CurIter.m_LastNode->getNextSibling() != NULL) {
        m_CurIter.m_LastNode = m_CurIter.m_LastNode->getNextSibling();
        return m_CurIter.m_LastNode->getObject();
      }
      
      // If we are here, we are on a rightmost leaf node. Pop the stack, and go from there.
      
      // If we haven't got any stack, then we are ont he rightmost root node, whihc is a leaf
      if (m_CurIter.m_TraverseStack.size() == 0) {
        return NULL;
      }
      
      
      // Get the node from the stack, and go there.
      m_CurIter.m_LastNode = m_CurIter.m_TraverseStack.back();;
      m_CurIter.m_TraverseStack.pop_back();
      return m_CurIter.m_LastNode->getObject();
    }
    
    return NULL;
  }
  
  MBaseObjectPtr MBaseObjectTree::getCurrent() {
    if (m_CurIter.m_LastNode != NULL)
      return m_CurIter.m_LastNode->getObject();
    
    return NULL;
  }
  
  MTreeObjectNodePtr MBaseObjectTree::getCurrentNode() {
    return m_CurIter.m_LastNode;
  }

  void MBaseObjectTree::setParent(MTreeObjectNodePtr Node, MTreeObjectNodePtr ParentNode) {
    if (Node == NULL)
      return;
    
    // Make sure we are not just setting it to its current parent.
    if (Node->getParent() == ParentNode)
      return;
    
    // Make sure we are not setting the node's parent to itself.
    if (Node == ParentNode)
      return;
    
    // Make sure the ParentNode is not a child of the current node.
    if (ParentNode != NULL) {
      if (ParentNode->isChildOf(Node))
        return;
    }
    
    // Remove the Node from its sibling list. Only if it has a parent
    if (Node->getParent() != NULL) {
      // Check to see if it is the first child of the current parent.
      if (Node->getParent()->getFirstChild() == Node)
      {
        Node->getParent()->setFirstChild(Node->getNextSibling());
        if (Node->getNextSibling() != NULL)
          Node->getNextSibling()->setPrevSibling(NULL);
        
        Node->setNextSibling(NULL);
        Node->setParent(NULL);
      }
      else
      {
        // If it is not the first child, then we can safely remove it.
        if (Node->getPrevSibling() != NULL)
          Node->getPrevSibling()->setNextSibling(Node->getNextSibling());
        
        if (Node->getNextSibling() != NULL)
          Node->getNextSibling()->setPrevSibling(Node->getPrevSibling());
        
        Node->setNextSibling(NULL);
        Node->setPrevSibling(NULL);
        Node->setParent(NULL);
      }
    } else {
      // Remove the Node from the root sibling list if it has no parent.
      
      // If it is the Root node, and it is being removed, we must update m_Root
      if (Node == m_Root) {
        if (Node->getNextSibling() != NULL)
          Node->getNextSibling()->setPrevSibling(NULL);
        
        m_Root = Node->getNextSibling();
      } else {
        // Just remove the node normally if it is not m_Root
        if (Node->getPrevSibling() != NULL)
          Node->getPrevSibling()->setNextSibling(Node->getNextSibling());
        
        if (Node->getNextSibling() != NULL)
          Node->getNextSibling()->setPrevSibling(Node->getPrevSibling());
      }
      
      Node->setNextSibling(NULL);
      Node->setPrevSibling(NULL);
      Node->setParent(NULL);
    }
    
    
    // Now Set the parent of the node.
    
    // Are we setting it to a root node?
    if (ParentNode == NULL) {
      if (m_Root != NULL)
        m_Root->setPrevSibling(Node);
      
      Node->setNextSibling(m_Root);
      m_Root = Node;
    } else {
      if (ParentNode->getFirstChild() != NULL)
        ParentNode->getFirstChild()->setPrevSibling(Node);
      
      Node->setNextSibling(ParentNode->getFirstChild());
      ParentNode->setFirstChild(Node);
      
      Node->setParent(ParentNode);
    }
  }
  
  void MBaseObjectTree::finishWithUndoNode() {
    nodeUndo = NULL;
    MBaseObject::finishWithUndoNode();
  }

  void MBaseObjectTree::ensureUndo() {
    if (undoEnabled) {
      doEnsureUndo(nodeUndo, this);
    }
  }

  MBaseObjectTree::Undo::Undo(MBaseObjectTree *node) {
    object = node;
    stored.m_Root = node->m_Root;
    stored.m_Count = node->m_Count;
  }

  MUndoableObject* MBaseObjectTree::Undo::getObject() {
    return &*object;
  }

  void MBaseObjectTree::Undo::undo() {
    std::swap(stored.m_Root, object->m_Root);
    std::swap(stored.m_Count, object->m_Count);
  }

  void MBaseObjectTree::Undo::redo() {
    std::swap(stored.m_Root, object->m_Root);
    std::swap(stored.m_Count, object->m_Count);
  }

  
  
  //----------------------------------------------------------------------------------------
  //  MSortedNumberList
  //----------------------------------------------------------------------------------------
  int      m_Num, m_NumAllocated;
  
  MSortedNumberList::MSortedNumberList()
  {
    m_Num = 0;
    m_NumAllocated = 0;
    m_Data = NULL;
  }
  
  MSortedNumberList::~MSortedNumberList() {
    removeAll();
  }
  
  void MSortedNumberList::removeAll() {
    if (m_Data) {
      delete[] m_Data;
    }
    
    m_Data = NULL;
    m_Num = 0;
    m_NumAllocated = 0;
  }
  
  int MSortedNumberList::add(int Num) {
    int   n;
    
    if (find(Num) != -1)
      return 0;
    
    // Find the location of the new element
    for (n = 0; n < m_Num && m_Data[n] < Num; n++) {
    }
    
    // n is now the location for the new element.
    
    insertAt(n, Num);
    
    return 1;
  }
  
  int MSortedNumberList::remove(int Num)
  {
    int Location, n;
    
    Location = find(Num);
    
    if (Location == -1)
      return 0;
    
    m_Num--;
    
    // Shift all the elements after it down one place
    for (n = Location; n < m_Num; n++)
    {
      m_Data[n] = m_Data[n+1];
    }
    
    return 1;
  }
  
  int MSortedNumberList::find(int Num)
  {
    // Just do a linear seach for now, is really slow, will change after to a binary search
    
    int   n;
    
    for (n = 0; n < m_Num; n++)
    {
      if (m_Data[n] == Num)
        return n;
    }
    return -1;
  }
  
  int MSortedNumberList::getItem(int Index)
  {
    if (Index >=0 && Index < m_Num)
    {
      return m_Data[Index];
    }
    
    return 0;
  }
  
  void MSortedNumberList::insertAt(int Index, int Value)
  {
    if (m_Num == 0)
    {
      m_Data = new int[MSORTEDLIST_GRAN];
      m_Num = 1;
      m_NumAllocated = MSORTEDLIST_GRAN;
      
      m_Data[0] = Value;
      
      return;
    }
    
    // Check if we need to expand the list size
    if (m_Num == m_NumAllocated)
    {
      int   *NewData;
      NewData = new int[m_NumAllocated + MSORTEDLIST_GRAN];
      memcpy(NewData, m_Data, m_Num * sizeof(int));
      delete[] m_Data;
      m_Data = NewData;
      
      m_NumAllocated += MSORTEDLIST_GRAN;
    }
    
    // Insert the element at the given location.
    // go backwards from the end, and shift elements along.
    int   n;
    
    for (n = m_Num; n > Index; n--)
    {
      m_Data[n] = m_Data[n-1];
    }
    
    m_Data[Index] = Value;
    m_Num++;
  }
 

}