#include "StdAfx.h"

#include "MBaseObject.h"
#include <stdlib.h>

#include "MListsTrees.h"
#include "MSystemManager.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {
  
  //----------------------------------------------------------------------------------------
  //  MBaseObject
  //----------------------------------------------------------------------------------------
  MBaseObject::MBaseObject() {
    m_CachedTime = 0xFFFFFFFF;
    flags.assignFlags(OBJECTFLAG_TIME_IS_INPUT);
  }
  
  MBaseObject::~MBaseObject() {
  }
  
  MBaseObjectPtr MBaseObject::createNew() {
    MBaseObjectPtr NewObj;
    
    NewObj = new MBaseObject();
    NewObj->setFrom(this);
    
    return NewObj;
  }
  
  void MBaseObject::setFrom(MBaseObjectPtr SrcObj) {
    flags = SrcObj->flags;
  }
  
  long MBaseObject::getTime() {
    MIntParameterPtr timeParam = MSystemManager::getTimeParameter();

    if (timeParam != NULL) {
      return timeParam->getValue();
    }
    return 0;
  }

  bool MBaseObject::updateObject() {
    bool result = true;

    // only do an object update if required.
    if (isFlagged(OBJECTFLAG_TIME_IS_INPUT)) {
      long time = getTime();
      if (m_CachedTime != time) {
        m_CachedTime = time;
        result = doUpdateObject();
      } else if (isFlagged(OBJECTFLAG_NEEDS_UPDATE)) {
        unsetFlag(OBJECTFLAG_NEEDS_UPDATE);
        result = doUpdateObject();
        // we unset the flag again, in case it has been reset during the course of the update process.
        unsetFlag(OBJECTFLAG_NEEDS_UPDATE);
      }
    } else if (isFlagged(OBJECTFLAG_NEEDS_UPDATE)) {
      result = doUpdateObject();
      unsetFlag(OBJECTFLAG_NEEDS_UPDATE);
    }
    return result;
  }

  bool MBaseObject::doUpdateObject() {
    return true;
  }

  void MBaseObject::setFlag(AztecFlags Flag) {
    if (!flags.isFlagged(Flag)) {
      if (Flag != OBJECTFLAG_NEEDS_UPDATE) {
        ensureFlagUndo();
      }
      flags.setFlag(Flag);
    }
  }

  bool MBaseObject::isFlagged(AztecFlags Flag) {
    return flags.isFlagged(Flag);
  }

  void MBaseObject::unsetFlag(AztecFlags Flag) {
    if (flags.isFlagged(Flag)) {
      if (Flag != OBJECTFLAG_NEEDS_UPDATE) {
        ensureFlagUndo();
      }
      flags.unsetFlag(Flag);
    }
  }

  void MBaseObject::toggleFlag(AztecFlags Flag) {
    if (Flag != OBJECTFLAG_NEEDS_UPDATE) {
      ensureFlagUndo();
    }
    flags.toggleFlag(Flag);
  }

  AztecFlags MBaseObject::getFlags() const {
    return flags.getFlags();
  }

  void MBaseObject::assignFlags(AztecFlags Flag) {
    if (flags.getFlags() != Flag) {
      if ((flags.getFlags() & ~OBJECTFLAG_NEEDS_UPDATE) != (Flag & ~OBJECTFLAG_NEEDS_UPDATE)) {
        ensureFlagUndo();
      }
      flags.assignFlags(Flag);
    }
  }

  void MBaseObject::finishWithUndoNode() {
    undo = NULL;
  }

  void MBaseObject::ensureFlagUndo() {
    doEnsureUndo(undo, this);
  }

  MBaseObject::Undo::Undo(const MBaseObjectPtr &obj) {
    object = obj;
    stored = obj->flags;
  }

  MUndoableObject* MBaseObject::Undo::getObject() {
    return &*object;
  }

  void MBaseObject::Undo::undo() {
    std::swap(stored, object->flags);
  }

  void MBaseObject::Undo::redo() {
    std::swap(stored, object->flags);
  }


  //---------------
  //  MObjectNode
  //---------------
  
  MObjectNode::MObjectNode() {
    m_Obj = NULL;
  }
  
  MObjectNode::~MObjectNode() {
    setObject(NULL);
  }
  
  
  MBaseObjectPtr MObjectNode::setObject(MBaseObjectPtr Obj) {
    MBaseObjectPtr OldObj;

    OldObj = m_Obj;
    m_Obj = Obj;

    return  OldObj;
  }
  
  MBaseObjectPtr MObjectNode::getObject() {
    return m_Obj;
  }

}
