// MSystemManager.h: interface for the MSystemManager class.
//
//////////////////////////////////////////////////////////////////////

#ifndef SYSTEMMANAGER_H
#define SYSTEMMANAGER_H

#include "ModelGeneric.h"

namespace Aztec {

  class MGlobalSettings;
  class MSystemManager;
  class MDllFile;

  typedef MRefCountedPtr<MGlobalSettings> MGlobalSettingsPtr;
  typedef MRefCountedPtr<MSystemManager> MSystemManagerPtr;
  typedef MRefCountedPtr<MDllFile> MDllFilePtr;
}

#include "MBaseObject.h"
#include "MListsTrees.h"
#include "LogFile.h"
#include "MUndoManager.h"
#include <MSceneRenderer.h>
#include <MPluginManager.h>
#include <translator/MImageTranslator.h>
#include <translator/MSceneTranslator.h>
#include <translator/MMaterialTranslator.h>

#include <map>

namespace Aztec {

  class MGlobalSettings : public MBaseObject {
  public:
    MStr        m_CFGFilename, m_KEYFilename;
    
    bool        m_Animate;
    
    MGlobalSettings();
    
    void ResetToDefaults();
    
    int ReadFilenamesFromReg();
    int SaveFilenamesToReg();
    
    int ReadConfig();
    int WriteConfig();
  };
  
  
  class MGENEXPORT MSystemManager : public MRefCountedObject {
  public:
    typedef std::vector<MImageTranslatorPtr> ImageTranslatorVector;

    // System Man functions
    static MSystemManagerPtr getInstance();
    static void cleanInstance();

    int findAndLoadDLLs(const MStr &directory, const MStr &FileMask, bool Init = true, bool RemoveOnFail = true);
    int loadDll(MStr Name, bool Init = true, bool RemoveOnFail = true);
    
    // Object class registering and what not
    MPluginManagerPtr getPluginManager();

    /// @deprecated Use the getPluginManager instead.
    void getTranslatorFromIndex(int Index, MSceneTranslatorPtr &Trans, bool addAllFileTypes = true);
    /// @deprecated Use the getPluginManager instead.
    void getTranslatorFromIndex(int Index, MImageTranslatorPtr &Trans);
    /// @deprecated Use the getPluginManager instead.
    void getTranslatorFromIndex(int Index, MMaterialTranslatorPtr &Trans);
    /// @deprecated Use the getPluginManager instead.
    MStr createSceneTranslatorFilterString(bool addAllFileTypes = true);
    /// @deprecated Use the getPluginManager instead.
    MStr createImageTranslatorFilterString();
    /// @deprecated Use the getPluginManager instead.
    MStr createMaterialTranslatorFilterString();

    /**
     * This loads an image from the given filename. The image type will try to
     * be deduced from the filename an the file itself, and then loaded. 
     *
     * The loaded image is then cached, so if the file is requested again, it 
     * is returned quickly. The cached image will not be used if:
     * <UL>
     *   <LI>If the file specified has had the date changed at all, or
     *   <LI> The force argument passed into this function is true.
     * </UL>
     *
     * @param Filename The image to load
     * @param force If this is true, the cached image will never be used. If it
     *              is false, then the cached image is used if the attributes
     *              of the file haven't changed.
     * @return The image loaded from the file.
     * @see MImage
     */
    MImagePtr loadImage(const MStr &Filename, bool force = false);
    
    /**
     * Sets the current global scene, so that getScene() returns Scene from now on.
     * @deprecated Use MScene::setGlobalScene instead.
     * @param Scene the new scene to use.
     * @see MScene
     */
    void setScene(MScenePtr Scene);

    /**
     * Gets the current global scene.
     * @deprecated Use MScene::getGlobalScene instead.
     * @see MScene
     */
    MScenePtr getScene();

    /**
     * Gets the object that is used to log text output to various feeds, such 
     * as a log file, the screen or something else entirely.
     * @see MLogFile
     */
    MLogFilePtr getLogger();

    /**
     * A quick way to log some text to the log files. Uses the same syntax as printf.
     */
    void logOutput(const char *FormatStr, ...);

    /**
     * Gets the settings that code can query to change behaviour according to 
     * a users preferences.
     * @see MGlobalSettings
     */
    MGlobalSettingsPtr getSettings();

    /**
     * This gets the Undo Manager.
     * @see MUndoManager
     */
    static MUndoManagerPtr getUndoManager();

    /**
     * This gets the integer parameter that is our current time.
     */
    static MIntParameterPtr getTimeParameter();

    MBaseObjectListPtr getDllList();

    /**
     * This retrieves a directory where the settings for Aztec which effect
     * all the users on the one machine are stored. Normally, the directory
     * is used as a place to store files containing default settings, or
     * plugin installation information, and things like that. The directory
     * returned may be writable only for those with enough permissions on
     * the directory, so any code wanting to create or change files in here
     * should be exrtact careful to handle errors gracefully.
     *
     * @return The global settings path. The directory returned will contain
     *         whatever file naming conventions, such as back/forward slashes,
     *         used by the underlying operating system. It will not have a
     *         traling slash
     */
    std::string getGlobalSettingsPath();

    /**
     * This gets the directory where the settings for aztec for the current
     * user logged on are kept. Things such as keyoard configuration or any
     * customisation are kept here.
     *
     * @return The user settings path. The directory returned will contain
     *         whatever file naming conventions, such as back/forward slashes,
     *         used by the underlying operating system. It will not have a
     *         traling slash.
     */
    std::string getUserSettingsPath();


    // MBaseObject methods
    MStr getClassName();
    MStr getParentClassName();
    

  protected:
    MSystemManager();
    ~MSystemManager();

    void initSystemManager();
    
    MBaseObjectListPtr m_DllList;

    class ImageListEntry {
    public:
      ImageListEntry(const MImagePtr &newImage, const std::string &newFilename);
      ImageListEntry(const ImageListEntry &src);

      MImagePtr getImage();

      std::string getFilename();

      /**
       * This checks the file attributes (date, size etc) and if they have
       * changed. Returns true if they have changed since the last time this
       * function was called, false if they haven't.
       */
      bool checkAttributes();

    private:
      /// This is the image that we are talking about
      MImagePtr image;

      /// The filename used to make the image.
      std::string filename;

      
#ifdef WIN32
      /**
       * This is the file attributes for the file when it was read 
       * last. This is used to determine if a file should be reloaded, or
       * if a cached image is sufficient.
       */
      WIN32_FILE_ATTRIBUTE_DATA lastAttributes;
#endif
    };

    typedef std::map<std::string, ImageListEntry> ImageMap;
    ImageMap imageList;
    
    MGlobalSettingsPtr m_SystemSettings;
    
    MLogFilePtr m_LogFile;

    /**
     * The undo manager for this sessios
     */
    MUndoManagerPtr m_UndoManager;

    /**
     * This is our global time parameter.
     */
    MIntParameterPtr m_TimeParam;

    MPluginManagerPtr pluginManager;

  };
  
  // class to handle a single dll file containing one plugin
  class MGENEXPORT MDllFile : public MBaseObject {
  protected:
  public:
    typedef enum {VERSIONFUNCNOTFOUND = -1, 
                  COUNTFUNCNOTFOUND = -2, 
                  REGISTERFUNCNOTFOUND = -3, 
                  VERSIONDIFF = -4,
                  SUCCEED = 0} DLLReturnValue;

    MDllFile();
    virtual ~MDllFile();
    
    int setFilename(MStr Name);
    int setSysManager(MSystemManagerPtr SysMan);
    int dllInit();
    int dllFinish();

    HINSTANCE getInstance();

    // MBaseObject methods
    virtual MStr getClassName();
    virtual MStr getParentClassName();

  protected:
    MSystemManagerPtr m_SysMan;
    
    HINSTANCE		m_DllInstance;
    MStr			m_Filename;
    
    int				m_NumClasses;
    
  };
  
  typedef int (*QueryVersionFunc)(long *Dest);
  typedef int (*QueryDescFunc)(char *Dest);
  typedef int (*QueryCountFunc)();
  typedef int (*RegisterFunc)(MSystemManager *SysMan, int IDNo);

  MGENEXPORT MSystemManagerPtr getSystemManager();
  void MGENEXPORT setSystemManager(MSystemManagerPtr SysMan);
  
}

#endif 
