#ifndef MParameterObjectList_Header
#define MParameterObjectList_Header

#include "ModelGeneric.h"

namespace Aztec {
  
  class MParameterObjectList;
  
  typedef MRefCountedPtr<MParameterObjectList> MParameterObjectListPtr;
}

#include "MDAGNode.h"
#include "MParameterObject.h"
#include <vector>

namespace Aztec {
  
  class MGENEXPORT MParameterObjectList : public MDAGNode {
  public:
    
    /**
     * This constructs a Parameter List with the given object
     * as our owner.
     *
     * @param owner The owner of this object list.
     */
    MParameterObjectList(MBaseObject *owner);
    ~MParameterObjectList();

    /**
     * Gets the MNamedobject that contains these parameters.
     */
    MBaseObject* getOwner() const;
    
    /**
     * This adds the given parameter to the parameter list. This 
     * will also create a graph connection between the owner of
     * this Parameter List and the parameter. 
     *
     * @param ParamObj The parameter to add
     * @return The number of parameters in this list.
     */
    int addParam(MParameterObjectPtr ParamObj);

    /**
     * This will replace a Parameter Object in this list with the parameter
     * provided. 
     */
    int replaceParamObject(const MStr &Name, MParameterObjectPtr Param);

    /**
     * This deletes all the parameters in this Parameter List.
     */
    void deleteAllParams();
    
    /**
     * This removes the given parameter from this parameter list.
     *
     * @param index The index of the parameter to remove.
     */
    void removeParameter(int index);

    /**
     * This retrieves the Parameter that has the given name.
     * It will first search the parameters for their short name, then by their
     * long name, followed by their friendly name. If no match could be found
     * then the function returns NULL.
     *
     * @param name The name of the parameter to find.
     * @return The Parameter object that was found. NULL if no parameter
     *         matching the given name was found.
     */
    MParameterObjectPtr getParameter(const MStr &name);

    /** 
     * This gets the parameter at the given index. If the index given
     * is out of bounds (less than zero, or greater than or equal to 
     * getNumParams()) then we return NULL.
     *
     * @param index The index of the parameter to retrieve.
     * @return The ParameterObject at the specified index. If the index is not
     *         valid, then NULL is returned.
     */
    MParameterObjectPtr getParameter(int index);
    
    /**
     * This gets the index of the given parameter with the given name. If
     * the parameter cannot be found in the List, then -1 is returned. This
     * uses the same searching method as the <code>getParameter(MStr)
     * </code> method.
     *
     * @param name The name of the parameter to search for.
     * @return The index of the parameter that has the given name. -1 if
     *         no parameter could be found.
     */
    int getParameterIndex(const MStr &name);

    /**
     * This gets the number of parameters contained in this parameter list.
     *
     * @return The number of parameters in this parameter list.
     */
    int getNumParams();
    
    /**
     * This copies the values from the source list into this one. For each
     * parameter in the source list, we attempt to set a parameter of the 
     * same name to the same value, using the parameter's 
     * <code>setFromParameter</code> method. If this list does not contain 
     * a parameter of the same name, then nothing takes place. 
     *
     * @param sourceList This is the list we are copying values from.
     * @return The number of parameters that were set.
     */
    int setFromList(MParameterObjectListPtr sourceList);

    /**
     * This is a helper method to make reading in parameters quicker
     * and storing parameters physically smaller in the files.
     */
    static void setTemporaryParamMapping(const std::vector<std::string> &params);

    /**
     * This is a helper method to make reading in parameters quicker
     * and storing parameters physically smaller in the files.
     */
    static void getTemporaryParamMapping(std::vector<std::string> &params);

    static std::string getClassNameFromTempID(int ID);

    bool doUpdateObject();


  protected:
    std::vector<MParameterObjectPtr> m_ParamList;

    static std::vector<std::string> tempMapping;

    /**
     * This is the owner of this parameter object.
     */
    MBaseObject *m_Owner;

    /**
     * This makes the connection between the parameter, this object
     * list, and our owner.
     */
    void makeConnection(MParameterObjectPtr parameter);

    /**
     * This breaks the connection between the parameter, this object
     * list, and our owner.
     */
    void breakConnection(MParameterObjectPtr parameter);
    
  };
  
  
}


#endif
