#ifndef MEditableComponentisedObject_Header
#define MEditableComponentisedObject_Header

#include <ModelGeneric.h>

namespace Aztec {
  class MEditableComponentisedObject;
  typedef MRefCountedPtr<MEditableComponentisedObject> MEditableComponentisedObjectPtr;
}

#include <MComponentisedObject.h>

namespace Aztec {
  //--------------
  //  MEditableComponentisedObject
  //--------------
  class MGENEXPORT MEditableComponentisedObject : public virtual MComponentisedObject {
  public:

    class TransformModifier {
    public:
      /**
       * This is called just before the transformComponents() method adjust a 
       * 3D coordinate. The original and new coordinate is passed in, and the 
       * overriding function can optionally alter the coordinates. Possible 
       * uses for this include 3d snapping of points.
       */
      virtual bool adjustTransform(const MVector3 &originalPos, MVector3 &newPos) = 0;
    };

    /**
     * This makes a copy of the positions of the given components that are
     * flagged, so that they can be restored later on. It is only 
     * required that one type of component be stored at any one time.
     *
     * @param type The type of component to store
     * @param flag The flags of the components to store. If this is 0, all
     *             components are stored.
     */
    virtual void storeComponentPositions(AztecFlags type, AztecFlags flag = 0) = 0;

    /**
     * This retrieves the copy of the positions of the given components 
     * that have been kept. It will only restore those components that
     * are current flagged, as long as those components have been previously
     * stored.
     *
     * @param type The type of component to restore
     * @param flag The flags of the components to restore. If this is 0, all
     *             components are stored.
     */
    virtual void restoreComponentPositions(AztecFlags type, AztecFlags flag = 0) = 0;

    /**
     * This applies a transformation matrix to a set of flagged components
     *
     * @param type The type of component
     * @param transform The transformation matrix to apply to the components.
     * @param flag The flag that each component must be flagged with in order
     *             to be effected. If this is zero, all components are 
     *             effected.
     * @param localTransform If this is true, the matrix is rotated to match
     *                       the local normal of the component, and then 
     *                       applied.
     */
    virtual void transformComponents(AztecFlags type, 
                                     const MMatrix4 &transform, 
                                     AztecFlags flag = 0, 
                                     bool localTransform = false,
                                     TransformModifier *modifier = NULL) = 0;

    /**
     * This tells the component object to update any animation information
     * from the current position of the components of the given type.
     * 
     * @param type The type of component to update
     * @param flag The flags that the component must have to update their 
     *             animation values. If this is zero, all the components are
     *             updated.
     * @param time The time to update the animation for.
     * @param createKey If this is true, then animation keys are set for the
     *                  required components. If this is false, then the
     *                  initial value of those animation tracks are set 
     *                  instead.
     */
    virtual void updateComponentAnimation(AztecFlags type, AztecFlags flag, long time, bool createKey) = 0;

    /**
     * This tells the component object to update any animation information
     * from the current position of the components of the given type at the 
     * given index.
     * 
     * @param type The type of component to update
     * @param index the index of the component to update.
     * @param time The time to update the animation for.
     * @param createKey If this is true, then animation keys are set for the
     *                  required components. If this is false, then the
     *                  initial value of those animation tracks are set 
     *                  instead.
     */
    virtual void updateSingleComponentAnimation(AztecFlags type, int index, long time, bool createKey) = 0;

  protected:
    MEditableComponentisedObject() { 
    }
    
    ~MEditableComponentisedObject() { 
    }

  };
}
#endif
