#ifndef MBASEOBJECT_H
#define MBASEOBJECT_H

#include "ModelGeneric.h"

namespace Aztec {
  
  class MBaseObject;
  typedef MRefCountedPtr<MBaseObject> MBaseObjectPtr;

}


#include "ModelTypes.h"
#include <MBAseUndoNode.h>

#define OBJECTFLAG_VISIBLE          (Aztec::AztecFlags)0x00000001
#define OBJECTFLAG_SELECTED         (Aztec::AztecFlags)0x00000002
#define OBJECTFLAG_NOCOMPONENTS     (Aztec::AztecFlags)0x00000008

// Flags for viewing the objects in tree views and channel boxes and stuff.
#define OBJECTFLAG_CHANNELEXPAND    (Aztec::AztecFlags)0x00000010
#define OBJECTFLAG_TREEEXPAND       (Aztec::AztecFlags)0x00000020    
#define OBJECTFLAG_MATERIALEXPAND   (Aztec::AztecFlags)0x00000040

#define OBJECTFLAG_NEEDS_UPDATE     (Aztec::AztecFlags)0x00000080
#define OBJECTFLAG_TIME_IS_INPUT    (Aztec::AztecFlags)0x00000100

#define OBJECTFLAG_TYPE_MESH        (Aztec::AztecFlags)0x00010000
#define OBJECTFLAG_TYPE_LIGHT       (Aztec::AztecFlags)0x00020000
#define OBJECTFLAG_TYPE_MATERIAL    (Aztec::AztecFlags)0x00040000
#define OBJECTFLAG_TYPE_MODIFIER    (Aztec::AztecFlags)0x00080000
#define OBJECTFLAG_TYPE_HELPER      (Aztec::AztecFlags)0x00100000
#define OBJECTFLAG_TYPE_CAMERA      (Aztec::AztecFlags)0x00200000
#define OBJECTFLAG_TYPE_SHAPE       (Aztec::AztecFlags)0x00400000
#define OBJECTFLAG_TYPE_ALL         (Aztec::AztecFlags)0xFFFF0000

#define OBJECTFLAG_ALLMODES         (Aztec::AztecFlags)0xFFFF0000

namespace Aztec {

  /**
   * This is the common base object used through out the model 
   * library. It contains lots of versatile important information 
   * about an object. Each BaseObject contains A 32 bit Flag, a list
   * of named prameters, and a 
   */
  class MGENEXPORT MBaseObject : public virtual MRefCountedObject, public MUndoableObject {
  public:
    DWORD    m_Temp;              // Temp number used in saving and loading of objects.
    
    // Construction/Destruction
    MBaseObject();
    virtual ~MBaseObject();
    
    // Class related
    virtual MStr getClassName() {return MStr("MBaseObject");};
    virtual MStr getParentClassName() {return MStr("");};
    virtual MBaseObjectPtr createNew();
    virtual void setFrom(MBaseObjectPtr SrcObj);

    /**
     * This gets the current time from the System manager. This is here 
     * for convinience.
     *
     * @return the current time that the system is at.
     */
    static long getTime();

    /**
     * This calls the doUpdateObject() method if an update is required.
     *
     * @return true if the update was successful.
     */
    bool updateObject();

    /**
     * This method does the actual updating of the object. Any derived
     * objects that need special processing when an update is required
     * MUST override this method.
     *
     * @return true if the update was successful.
     */
    virtual bool doUpdateObject();
    
    // Flag methods
    void setFlag(AztecFlags Flag);
    bool isFlagged(AztecFlags Flag);
    void unsetFlag(AztecFlags Flag);
    void toggleFlag(AztecFlags Flag);
    AztecFlags getFlags() const;
    void assignFlags(AztecFlags Flag);

    // MUndoableObject methods
    void finishWithUndoNode();

  private:
    /**
     * This is the time when the last update object occurred. 
     * If the system time is different to this time, then we need
     * an update.
     */
    long m_CachedTime;

    MFlagObject flags;

    class Undo : public MBaseUndoNode {
    public:
      Undo(const MBaseObjectPtr &obj);

      MUndoableObject* getObject();
      void undo();
      void redo();

    private:
      MFlagObject stored;
      MBaseObjectPtr object;
    };

    MRefCountedPtr<Undo> undo;

    void ensureFlagUndo();

    friend class Undo;
  };
  
  
  //----------------------------------------------------------------------------------------
  //  MObjectNode
  //----------------------------------------------------------------------------------------
  class MGENEXPORT MObjectNode : public MRefCountedObject, public MFlagObject {
  protected:
    MBaseObjectPtr m_Obj;
  public:
    MObjectNode();
    virtual ~MObjectNode();
    
    MBaseObjectPtr setObject(MBaseObjectPtr Obj);
    MBaseObjectPtr getObject();
  };
  
  
}


#endif
