#include <AztecGUICommonPCH.h>
#include <gui/MMenuBar.h>

#include <gui/MApplication.h>

#include <assert.h>

static const int TEXT_MARGIN_WIDTH = 3;
static const int TEXT_MARGIN_HEIGHT = 4;

static const int MB_POPUPMENU = WM_USER + 1101;

// These values were taken from WinUser.h from the august 2002 Platform SDK.
// note the wrapping #ifndef .. #endif were added manually.
#ifndef COLOR_MENUHILIGHT
#define COLOR_MENUHILIGHT       29
#endif
#ifndef COLOR_MENUBAR
#define COLOR_MENUBAR           30
#endif
#ifndef SPI_GETFLATMENU
#define SPI_GETFLATMENU                     0x1022
#endif



// These were taken from the winuser.h

extern "C" {

WINUSERAPI
BOOL
WINAPI
TrackMouseEvent(
    LPTRACKMOUSEEVENT lpEventTrack);

}

#if(_WIN32_WINNT < 0x0500)
#define TPM_NOANIMATION     0x4000L
#endif /* _WIN32_WINNT >= 0x0500 */


namespace Aztec {



  MMenuBar::MMenuBar() {
    mouseInside = false;
    lastX = 0;
    lastY = 0;
    hilightedMenu = -1;
    itemHeight = -1;
    openingMenus = false;
    constantSize = false;
    lastRowCount = 1;
    hilight3D = true;
    centreText = true;
    isMenuBar = true;//menuBar;
    useXPColours = false;

    if (isMenuBar) {
      constantSize = false;
      hilight3D = true;
      centreText = true;
    } else {
      constantSize = true;
      hilight3D = false;
      centreText = false;
    }

    // find out if we are using windows xp, and if we are, enable a check
    // for using flat menus
    OSVERSIONINFO version;
    version.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
    ::GetVersionEx(&version);

    if ((version.dwMajorVersion = 5 && version.dwMinorVersion > 1) ||
      version.dwMajorVersion > 5) 
    {
      useXPColours = true;
      BOOL flat = FALSE;
      SystemParametersInfo(SPI_GETFLATMENU, 0, &flat, 0);
      hilight3D = !flat;
    }

    hilight3D = false;

  }

  MMenuBar::~MMenuBar() {
  }


  MMenuBarPtr		g_pMenuBar	= NULL;
  HHOOK			g_hMsgHook	= NULL;
  ////////////////////////////////////////////////////////////////////////////////
  //	The hook, used to process message when menu is visible
  LRESULT CALLBACK MenuInputFilter(int nCode, WPARAM wParam, LPARAM lParam)
  {
	  MSG* pMsg = (MSG*)lParam;

	  if(g_pMenuBar == NULL || nCode != MSGF_MENU)
		  return CallNextHookEx(g_hMsgHook,nCode,wParam,lParam);
	  if(g_pMenuBar->onMenuInput(pMsg))
		  return TRUE;
	  else
		  return CallNextHookEx(g_hMsgHook,nCode,wParam,lParam);
  }

  bool MMenuBar::wndProc(UINT message, WPARAM wParam, LPARAM lParam, LRESULT *result) {
    // we call the componnet message handler first so it does any speical 
    // dispatching before we handle our own messages.
    bool handled = MComponent::wndProc(message, wParam, lParam, result);
    
    if (message == WM_LBUTTONDOWN) {
      // check to see if we have clicked on a menu item, if we have, then
      // we switch to the mode where we open menus up.
      int xPos = LOWORD(lParam);  // horizontal position of cursor
      int yPos = HIWORD(lParam);  // vertical position of cursor
      int index = getItemFromPos(xPos, yPos);
      
      setHilightedMenu(index, true);
      
      //    return 1;
    }
    if (message == WM_LBUTTONUP) {
      int xPos = LOWORD(lParam);  // horizontal position of cursor
      int yPos = HIWORD(lParam);  // vertical position of cursor
      int index = getItemFromPos(xPos, yPos);
      
      setHilightedMenu(index, false);
    }
    
    if (message == WM_MOUSEMOVE) {
      int xPos = LOWORD(lParam);  // horizontal position of cursor
      int yPos = HIWORD(lParam);  // vertical position of cursor
      int index = getItemFromPos(xPos, yPos);

      if (!mouseInside) {
        TRACKMOUSEEVENT mouseEvent;
        mouseEvent.cbSize = sizeof(TRACKMOUSEEVENT);
        mouseEvent.dwFlags = TME_LEAVE;
        mouseEvent.hwndTrack = m_hWnd;
        mouseEvent.dwHoverTime = 0;
        ::TrackMouseEvent(&mouseEvent);
        mouseInside = true;
      }
      
      setHilightedMenu(index);

    }

    if (message == MB_POPUPMENU) {
      setHilightedMenu(wParam);
    }
    if (message == WM_MOUSELEAVE) {
      if (g_hMsgHook == NULL && !openingMenus)  {
        setHilightedMenu(-1, false);
        mouseInside = false;
      }
    }

    return handled;
  }

  BOOL MMenuBar::onMenuInput(MSG* pMsg)
  {
	  BOOL bResult = FALSE;

	  switch(pMsg->message)
	  {
	  case WM_MOUSEMOVE:
		  {
			  POINT pt;
			  pt.x = LOWORD(pMsg->lParam);
			  pt.y = HIWORD(pMsg->lParam);

        if (pt.x != lastX || pt.y != lastY) {

          lastX = pt.x;
          lastY = pt.y;

          ::ScreenToClient(m_hWnd, &pt);

			    int nTest = getItemFromPos(pt.x, pt.y);

			    if(nTest >= 0 && nTest < items.size() && nTest != hilightedMenu)
			    {
				    SendMessage(m_hWnd, WM_CANCELMODE,0,0);
				    PostMessage(m_hWnd, MB_POPUPMENU,nTest,1);
				    bResult = TRUE;
			    }

        }
		  }
		  break;
	  case WM_LBUTTONDOWN:
		  {
			  POINT pt;
			  pt.x = LOWORD(pMsg->lParam);
			  pt.y = HIWORD(pMsg->lParam);
        ScreenToClient(m_hWnd, &pt);

			  int nTest = getItemFromPos(pt.x, pt.y);

			  if(nTest<0)
          setHilightedMenu(-1, false);
			  else if(nTest == hilightedMenu)
			  {
          setHilightedMenu(-1, false);
				  PostMessage(m_hWnd, WM_CANCELMODE,0,0);
				  bResult = TRUE;
			  }
		  }
		  break;
	  case WM_KEYDOWN:
		  {
			  TCHAR vkey = pMsg->wParam;
			  if(vkey == VK_LEFT)
			  {
				  PostMessage(m_hWnd, WM_CANCELMODE,0,0);
				  PostMessage(m_hWnd, MB_POPUPMENU,(hilightedMenu + items.size() - 1) % items.size(),1);
				  bResult = TRUE;
			  }
			  else if(vkey == VK_RIGHT)
			  {
				  PostMessage(m_hWnd, WM_CANCELMODE,0,0);
				  PostMessage(m_hWnd, MB_POPUPMENU,(hilightedMenu + 1) % items.size(),1);
				  bResult = TRUE;
			  }
			  else if (vkey == VK_ESCAPE)
			  {
				  PostMessage(m_hWnd, WM_CANCELMODE,0,0);
          setHilightedMenu(-1, false);
				  bResult = TRUE;
			  }
		  }
		  break;
	  case WM_MENUSELECT:
//		  getParent()->SendMessage(m_hWnd, WM_MENUSELECT, pMsg->wParam, pMsg->lParam);
		  bResult = TRUE;
		  break;
	  default:
		  break;
	  }

	  return bResult;
  }

    // MComponentImpl methods
  bool MMenuBar::createImpl() {
    MApp *app;
    HWND parentHWND = NULL;

    if (m_Parent != NULL) {
      parentHWND = m_Parent->getHWnd();
    }

    app = MApp::getInstance();

    m_hWnd = ::CreateWindow((LPCTSTR)app->getWindowClass("MMenuBar"), NULL, WS_CHILD,
                            0, 10, 0, 10, parentHWND, NULL, app->getHInstance(), NULL);

    if (m_hWnd != 0) {
      app->registerComponent(m_hWnd, this);

      onCreate();

      setVisible(true);
      ::UpdateWindow(m_hWnd);

      return true;
    }

    return false;
  }

  static void draw3dRect(HDC dc, const RECT &rect, bool sunken = false) {
    HPEN hPen, hOldPen;
    COLORREF lightColour = ::GetSysColor(COLOR_3DHILIGHT);
    COLORREF darkColour = ::GetSysColor(COLOR_3DSHADOW);
    COLORREF topLeftColour = sunken ? darkColour : lightColour;
    COLORREF bottomRightColour = sunken ? lightColour : darkColour;
    
    hPen = ::CreatePen(PS_SOLID, 1, topLeftColour);
    hOldPen = (HPEN)::SelectObject(dc, hPen);
    ::MoveToEx(dc, rect.right, rect.top, NULL);
    ::LineTo(dc, rect.left, rect.top);
    ::LineTo(dc, rect.left, rect.bottom);
    ::SelectObject(dc, hOldPen);
    ::DeleteObject(hPen);
    
    hPen = ::CreatePen(PS_SOLID, 1, bottomRightColour);
    hOldPen = (HPEN)::SelectObject(dc, hPen);
    ::MoveToEx(dc, rect.right, rect.top, NULL);
    ::LineTo(dc, rect.right, rect.bottom);
    ::LineTo(dc, rect.left, rect.bottom);
    ::SelectObject(dc, hOldPen);
    ::DeleteObject(hPen);
  }
 
  bool MMenuBar::onPaint() {
    HDC         dc;
    HBITMAP     bitmap, oldBitmap;
    
    RECT        ClientRect;
    HFONT       hOldFont;
    
    ::GetClientRect(m_hWnd, &ClientRect);
    
    dc = ::CreateCompatibleDC(paintDC);
    bitmap = ::CreateCompatibleBitmap(paintDC, ClientRect.right,
      ClientRect.bottom);
    oldBitmap = (HBITMAP)::SelectObject(dc, bitmap);
    
    hOldFont = (HFONT)::SelectObject(dc, GetStockObject(DEFAULT_GUI_FONT));
    
    HBRUSH menuBackground = ::GetSysColorBrush(COLOR_MENUBAR);
    
    // If that failed, just use the standard 3d colour.
    if (menuBackground == NULL) {
      menuBackground = ::GetSysColorBrush(COLOR_MENU);
    }
    if (menuBackground == NULL) {
      menuBackground = ::GetSysColorBrush(COLOR_3DFACE);
    }
    // fill in the menu bar
    ::FillRect(dc, &ClientRect, menuBackground);
    
    // draw the menu items.
    drawMenuItems(dc);
    
    // revert the dc back to its original state.
    ::SelectObject(dc, hOldFont);
    
    // copy the bitmap to the screen dc
    ::GetClientRect(m_hWnd, &ClientRect);
    
    ::BitBlt(paintDC, 0,0,ClientRect.right, ClientRect.bottom, dc, 0,0,SRCCOPY);
    
    ::SelectObject(dc, oldBitmap);
    ::DeleteObject(bitmap);
    ::DeleteDC(dc);

    return true;
  }

  MSize2D MMenuBar::getMinimumSize() {
    return MSize2D(32, getMenuHeight());
  }

  bool MMenuBar::isMinimumSizeSet() {
    return true;
  }

    // MMenuShell methods
  int MMenuBar::getItemCount() {
    return items.size();
  }

  MMenuItemPtr MMenuBar::getItem(int index) {
    return items[index].menuItem;
  }

  void MMenuBar::addItem(MMenuItemPtr item) {
    addItem(item, getItemCount());
  }

  void MMenuBar::addItem(MMenuItemPtr item, int index) {
    MenuItem newItem;
    newItem.menuItem = item;
    // we do a getHMenu here to force it to create a popup menu instead of a normal menu.
    if (item->getSubMenu() != NULL) {
      item->getSubMenu()->getHMenu(true);
    }
    items.insert(items.begin() + index, newItem);
  }

  void MMenuBar::addSeparator() {
    // do nothing
  }

  void MMenuBar::addSeparator(int index) {
    // do nothing
  }

  void MMenuBar::removeItem(int index) {
    items.erase(items.begin() + index);
  }

  bool MMenuBar::onItemClick(const MMenuItemPtr &item) {
    return false;
  }

  static COLORREF toColor(const MColour &col) {
    return RGB(col.getRuchar(), col.getGuchar(), col.getBuchar());
  }

  static void drawHilightFilled(HDC dc, RECT *rect, COLORREF colour) {
    HBRUSH brush = ::CreateSolidBrush(colour);
    HBRUSH oldBrush;

    MColour hilight((float)GetRValue(colour) / 255.0, 
                (float)GetGValue(colour) / 255.0, 
                (float)GetBValue(colour) / 255.0);

    HBRUSH menuBackground = ::GetSysColorBrush(COLOR_MENUBAR);
    COLORREF bgColour = ::GetSysColor(COLOR_MENUBAR);
    
    // If that failed, just use the standard 3d colour.
    if (menuBackground == NULL) {
      menuBackground = ::GetSysColorBrush(COLOR_MENU);
      bgColour = ::GetSysColor(COLOR_MENU);
    }
    if (menuBackground == NULL) {
      menuBackground = ::GetSysColorBrush(COLOR_3DFACE);
      bgColour = ::GetSysColor(COLOR_3DFACE);
    }


    MColour bg((float)GetRValue(bgColour) / 255.0, 
                (float)GetGValue(bgColour) / 255.0, 
                (float)GetBValue(bgColour) / 255.0);


    MColour half = 0.75 * bg + 0.25 * hilight;

    HPEN pen = ::CreatePen(PS_SOLID, 1, toColor(half));
    HPEN oldPen;


    oldBrush = (HBRUSH)::SelectObject(dc, brush);
    oldPen = (HPEN)::SelectObject(dc, pen);

    ::RoundRect(dc, rect->left, rect->top, rect->right, rect->bottom, 7, 7);

    ::SelectObject(dc, oldBrush);
    ::SelectObject(dc, oldPen);
    ::DeleteObject(brush);
  }

  void MMenuBar::drawMenuItems(HDC dc) {

    layoutItems(dc);

    HFONT oldFont, menuFont;

    SetBkMode(dc, TRANSPARENT);

    // TODO: get the actual menu font instead of the default one
    menuFont = (HFONT)::GetStockObject(DEFAULT_GUI_FONT);
    oldFont = (HFONT)::SelectObject(dc, menuFont);

    ::SetTextColor(dc, ::GetSysColor(COLOR_MENUTEXT));
    for (int index = 0; index < items.size(); ++index) {
      if (!hilight3D) {
        // if the item is the selected one, draw a coloured in box behind it
        if (hilightedMenu == index) {
          drawHilightFilled(dc, &items[index].rect, ::GetSysColor(useXPColours ? COLOR_MENUHILIGHT : COLOR_HIGHLIGHT)); 

          /**
          HBRUSH brush = ::CreateSolidBrush(::GetSysColor(useXPColours ? COLOR_MENUHILIGHT : COLOR_HIGHLIGHT));
          ::FillRect(dc, &items[index].rect, brush);
          ::DeleteObject(brush);
          */

          ::SetTextColor(dc, ::GetSysColor(COLOR_HIGHLIGHTTEXT));
        } else {
          ::SetTextColor(dc, ::GetSysColor(COLOR_MENUTEXT));
        }
      }

      DWORD flags;
      RECT rect = items[index].rect;
      if (centreText) {
        flags = DT_SINGLELINE | DT_CENTER | DT_VCENTER;
      } else {
        flags = DT_SINGLELINE | DT_VCENTER;
        rect.left += TEXT_MARGIN_WIDTH;
      }

      ::DrawText(dc, getCaption(index).c_str(), getCaption(index).length(), &rect, flags);
    }

    if (hilight3D) {
      if (hilightedMenu != -1) {
        RECT border = items[hilightedMenu].rect;
        draw3dRect(dc, border, openingMenus);
      }
    }

    ::SelectObject(dc, oldFont);

  }


  void MMenuBar::layoutItems(HDC dc) {
    calcItemSizes(dc);
    
    RECT clientRect;
    RECT itemRect;
    
    ::GetClientRect(m_hWnd, &clientRect);
    
    // give the menu bar a border
    clientRect.left += 1;
    clientRect.right -= 1;
    clientRect.top += 1;
    clientRect.bottom -= 1;
    
    itemRect.left = clientRect.left;
    itemRect.top = clientRect.top;
    
    int rowCount = 1;
    for (ItemVector::iterator it = items.begin(); it != items.end(); ++it) {
      
      itemRect.right = itemRect.left + it->width;
      itemRect.bottom = itemRect.top + it->height;
      
      // if we extend beyond the edge of the window, then we should wrap around.
      if (itemRect.right > clientRect.right) {
        itemRect.right += clientRect.left - itemRect.left;
        itemRect.left = clientRect.left;
        itemRect.top += itemHeight + 1;
        itemRect.bottom += itemHeight + 1;
        ++rowCount;
      }
      
      it->rect = itemRect;
      
      itemRect.left = itemRect.right + 1;
    }
    
    if (lastRowCount != rowCount) {
      lastRowCount = rowCount;
      RECT windowRect;
      ::GetWindowRect(m_hWnd, &windowRect);
      int width = windowRect.right - windowRect.left;

      MContainerPtr cont = getParent();

      while (cont->getParent() != NULL) {
        cont = cont->getParent();
      }

      cont->doLayout();
    }
  }

  static MSize2D getTextSize(HDC dc, const std::string &str) {
    SIZE size;
    ::GetTextExtentPoint(dc, str.c_str(), str.length(), &size);
    
    size.cx += TEXT_MARGIN_WIDTH * 2;
    size.cy += TEXT_MARGIN_HEIGHT;

    return MSize2D(size.cx, size.cy);
  }

  void MMenuBar::calcItemSizes(HDC dc) {
    // first make sure our items matches up with our menu

    bool needToReleaseDC = false;
    if (dc == NULL) {
      dc = ::GetDC(m_hWnd);
      needToReleaseDC = true;
    }
  
    HFONT oldFont, menuFont;
  
    // TODO: get the actual menu font instead of the default one
    menuFont = (HFONT)::GetStockObject(DEFAULT_GUI_FONT);
    oldFont = (HFONT)::SelectObject(dc, menuFont);
  
    int largestWidth = 0;
    itemHeight = 0;

    // If we have items, use those to calculate the height, otherwise use some text.
    if (items.size() > 0) {

      for (int i = 0; i < items.size(); ++i) {
        MSize2D size = getTextSize(dc, getCaption(i));
    
        items[i].width = size.getWidth();
        items[i].height = size.getHeight();
    
        if (size.getHeight() > itemHeight) {
          itemHeight = size.getHeight();
        }
    
        if (size.getWidth() > largestWidth) {
          largestWidth = size.getWidth();
        }
      }

    } else {
      MSize2D size = getTextSize(dc, "abwqABWQ");
      itemHeight = size.getHeight();
      largestWidth = size.getWidth();
    }
    
    // if we want all our menu items to be of a constant size, then we go over
    // them and reassign the size.
    if (constantSize) {
      for (ItemVector::iterator it = items.begin(); it != items.end(); ++it) {
        it->width = largestWidth;
      }    
    }
  
    // restore the old font for the dc
    ::SelectObject(dc, oldFont);
  
    if (needToReleaseDC) {
      ::ReleaseDC(m_hWnd, dc);
    }
  }

  void MMenuBar::setHilightedMenu(int index, int changeOpeningMenu) {
    bool newOpenMenuValue = openingMenus;
    if (changeOpeningMenu == 0) {
      newOpenMenuValue = false;
    } else if (changeOpeningMenu == 1) {
      newOpenMenuValue = true;
    }

    if (hilightedMenu != index || openingMenus != newOpenMenuValue) {
      openingMenus = newOpenMenuValue;

      if (!(openingMenus && index == -1)) {
        hilightedMenu = index;
        
        if (m_hWnd != NULL) {
          ::InvalidateRect(m_hWnd, NULL, FALSE);
        }

        // if we are in open menu mode, we have to open its popup menu
        if (openingMenus && index != -1) {
          showMenu(index);
        } else {
          showMenu(-1);
        }
      }
    }


  }

  int MMenuBar::getItemFromPos(int x, int y) {
    for (int index = 0; index < items.size(); ++index) {
      if (x >= items[index].rect.left &&
          x <= items[index].rect.right &&
          y >= items[index].rect.top && 
          y <= items[index].rect.bottom) 
      {
        return index;
      }
    }

    return -1;
  }

  void ClientToScreen(HWND wnd, RECT *rect) {
    POINT a;

    a.x = rect->left;
    a.y = rect->top;
    ::ClientToScreen(wnd, &a);
    rect->left = a.x;
    rect->top = a.y;

    a.x = rect->right;
    a.y = rect->bottom;
    ::ClientToScreen(wnd, &a);
    rect->right = a.x;
    rect->bottom = a.y;
  }

  void MMenuBar::showMenu(int index) {
    
    if (index >= 0) {
      RECT menuRect;
      
      if (isMenuBar) {
        menuRect.left = items[index].rect.left;
        menuRect.top = items[index].rect.bottom + 1;
      } else {
        menuRect.left = items[index].rect.right;
        menuRect.top = items[index].rect.top - 1;
      }
      
      menuRect.right = menuRect.left + 50;
      menuRect.bottom = menuRect.top + 50;
      
      ClientToScreen(m_hWnd, &menuRect);
      
      MMenuPtr sub = items[index].menuItem->getSubMenu();

      HMENU currentMenu = sub != NULL ? sub->getHMenu(true) : NULL;

      //Install the hook
      g_pMenuBar = this;
      g_hMsgHook = SetWindowsHookEx(WH_MSGFILTER,
        MenuInputFilter, NULL, GetCurrentThreadId());
      
      // Show the menu
      BOOL result = TrackPopupMenuEx(
        currentMenu,
        TPM_LEFTALIGN | TPM_LEFTBUTTON | TPM_RIGHTBUTTON | TPM_VERTICAL | TPM_NOANIMATION, 
        menuRect.left, 
        menuRect.top, 
        m_hWnd, 
        NULL);
      
      DWORD err = 0;
      if (result == FALSE) {
        err = ::GetLastError();
      }
      
      // Uninstall the hook
      UnhookWindowsHookEx(g_hMsgHook);
      
      g_hMsgHook = NULL;
      g_pMenuBar = NULL;
      
//      items[index].finishedWithMenu(currentMenu);
      
    }
    
  }



  std::string MMenuBar::getCaption(int index) {
    return items[index].menuItem->getCaption();
  }

  int MMenuBar::getMenuHeight() {
    if (itemHeight == -1) {
      calcItemSizes(NULL);
    }
    return (itemHeight + 2) * lastRowCount + 1;
  }


  MMenuBar::MenuItem::MenuItem() {
    rect.left = 0;
    rect.right = 0;
    rect.top = 0;
    rect.bottom = 0;
    width = 0;
    height = 0;
  }

  MMenuBar::MenuItem::MenuItem(const MenuItem &src) {
    rect = src.rect;
    width = src.width;
    height = src.height;
    menuItem = src.menuItem;
  }

  MMenuBar::MenuItem::~MenuItem() {
  }


}

