/*

  Copyright (c) 2003 Phillip Martin
  Copyright (c) 2003 Richard van Eijbergen <richardve@users.sourceforge.net>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <AztecGUICommonPCH.h>
#include <gui/MComponent.h>

#include <gui/qt3/MAppImpl.h>
#include <gui/qt3/MContainerImpl.h>

#include <iostream>

#include <gui/qt3/QComponent_.moc>


namespace Aztec {

  MComponent::MComponent()
    : m_Handle(0l),
      m_Parent(0l) {

    m_MinimumSize.setWidth(-1);
    m_MinimumSize.setHeight(-1);
    m_PreferredSize.setWidth(-1);
    m_PreferredSize.setWidth(-1);
  }

  MComponent::MComponent(const std::string &name) 
    : m_Name(name),
      m_Handle(0l),
      m_Parent(0l) {

    m_MinimumSize.setWidth(-1);
    m_MinimumSize.setHeight(-1);
    m_PreferredSize.setWidth(-1);
    m_PreferredSize.setHeight(-1);
  }

  MComponent::~MComponent() {
    m_Parent = 0l;

    if (m_Handle) {
      delete m_Handle;
      m_Handle = 0l;
    }
  }

  bool MComponent::createImpl() {

    MApp* app = MApp::getInstance();
    if (m_Handle != 0l) return true;

    QWidget* parentWnd = 0l;
    if (m_Parent != 0l) parentWnd = m_Parent->getHandle();

    m_Handle = new CQComponent_(parentWnd, this);

    if (m_Handle != 0l) {
      app->registerComponent(m_Handle, this);

      onCreate();
      return true;
    }

    return false;
  }

  
  // MComponent methods.
 
  void MComponent::setName(const std::string name) {
    m_Name = name;
  }

  std::string MComponent::getName() {
    return m_Name;
  }

  void MComponent::setParent(MContainerPtr parent) {
    if (m_Parent == &*parent/* || m_Handle == 0l*/) return;

    m_Parent = &*parent;
    if (parent != 0l) parent->addComponent(this);

    // Create the physical Qt implementation if needed
    createImpl();

    QWidget* parentWnd = 0l;
    if (parent != 0l) parentWnd = parent->getHandle();

    m_Handle->reparent(parentWnd, QPoint(0, 0));
  }

  MContainerPtr MComponent::getParent() {
    return m_Parent;
  }

  void MComponent::setVisible(bool visible) {
    if (m_Handle == 0l) return;

    if (visible) {
      /**
       * We're testing the widget for its visibility here
       * because the main window will flash when trying
       * to show an already shown (child) window.
       * (seems to be a bug in XFree86 as Gtk suffers from
       * the same issue)
       */
      if (!m_Handle->isVisible()) m_Handle->show();
    }
    else {
      m_Handle->hide();
    }
  }


  void MComponent::setPosition(const MPoint2D &pos) {
    if (m_Handle == 0l) return;

    int w = m_Handle->width();
    int h = m_Handle->height();
    m_Handle->setGeometry(pos.x+1, pos.y+1, w, h);
  }

  void MComponent::setPosition(int X, int Y) {
    if (m_Handle == 0l) return;

    int w = m_Handle->width();
    int h = m_Handle->height();
    m_Handle->setGeometry(X+1, Y+1, w, h);
  }

  void MComponent::setPosition(int X, int Y, int width, int height) {
    if (m_Handle == 0l) return;
    m_Handle->setGeometry(X+1, Y+1, width/**/, height/**/);
  }

  MPoint2D MComponent::getPosition() {
    MRect2D rect = getWindowRect();
    return MPoint2D(rect.x1-1, rect.y1-1);
  }

  void MComponent::setSize(const MSize2D &size) {
    if (m_Handle == 0l) return;
    m_Handle->setGeometry(size.getLeft()+1, size.getTop()+1, size.getWidth()/**/, size.getHeight()/**/);
  }

  void MComponent::setSize(int width, int height) {
    if (m_Handle == 0l) return;

    int x1 = m_Handle->x();
    int y1 = m_Handle->y();

    /**
     * The win32 implementation shows that 0,0 ought to be used for the position.
     * However, this is somewhat misleading as the position arguments passed to
     * the win32 API are not used (see SWP_NOMOVE).
     * This is confusing when doing a non-win32 implementation, so keep an eye
     * on this when starting other implementations (ie. for OS X or so).
     */
    m_Handle->setGeometry(x1, y1, width/**/, height/**/);
  }

  MSize2D MComponent::getSize() {
    return getWindowRect();
  }

  MSize2D MComponent::getClientSize() {
    if (m_Handle == 0l) return MSize2D(0, 0, 10, 10);

    // TODO: More work? (see win32 impl)

    QRect rc = m_Handle->rect();
    return MSize2D(rc.left()-1, rc.top()-1, rc.right()/**/, rc.bottom()/**/);

    /*QRect rc = m_Handle->geometry();

    QPoint topLeft(rc.left(), rc.top());
    QPoint botRight(rc.right(), rc.bottom());

    topLeft = m_Handle->mapFromGlobal(topLeft);
    botRight = m_Handle->mapFromGlobal(botRight);

    return MSize2D(topLeft.x(), topLeft.y(), botRight.x(), botRight.y());*/
  }

  MSize2D MComponent::getMinimumSize() {
    if (isMinimumSizeSet()) {
      return m_MinimumSize;
    }

    // Otherwise return a tiny rectangle
    return MSize2D(10, 10);
  }

  void MComponent::setMinimumSize(const MSize2D &size) {
    m_MinimumSize = size;

    if (m_Handle != 0l) {
      m_Handle->setMinimumSize(QSize(size.getWidth(), size.getHeight()));
    }
  }

  bool MComponent::isMinimumSizeSet() {
    if (m_MinimumSize.getWidth() < 0 || m_MinimumSize.getHeight() < 0 ) {
      return false;
    }

    return true;
  }

  MSize2D MComponent::getPreferredSize() {
    if (isPreferredSizeSet()) {
      return m_PreferredSize;
    }

    // Otherwise return out minimum size
    return getMinimumSize();
  }

  void MComponent::setPreferredSize(const MSize2D &size) {
    m_PreferredSize = size;
  }

  bool MComponent::isPreferredSizeSet() {
    if (m_PreferredSize.getWidth() < 0 || m_PreferredSize.getHeight() < 0 ) {
      return false;
    }

    return true;
  }

  MRect2D MComponent::getWindowRect() {
    // TODO: More work? (see win32 impl)
    QRect rc = m_Handle->geometry();
    return MSize2D(rc.left()-1, rc.top()-1, rc.right()/**/, rc.bottom()/**/);

//    return MRect2D(0,0,20,20);
  }

  void MComponent::setWindowRect(const MRect2D &rect) {
    if (m_Handle == 0l) return;
    m_Handle->setGeometry(rect.x1+1, rect.y1+1, (rect.x2-rect.x1)/**/, (rect.y2-rect.y1)/**/);
  }

  void MComponent::refresh() {
    // TODO: Tests, what do we really need?
    if (m_Handle != 0l) {
      m_Handle->update();
      m_Handle->updateGeometry();
      //m_Handle->repaint(false);
    }
  }

  void MComponent::setEnabled(bool enabled) {
    if (m_Handle != 0l) {
      m_Handle->setEnabled(enabled);
    }
  }

  void MComponent::setMouseCapture(bool capturing) {
    /**
     * Commented out because Qt already
     * grabs the mouse when necessary
     */
    /*if (capturing) {
      m_Handle->grabMouse();
    } else {
      m_Handle->releaseMouse();
    }*/
  }

  void MComponent::setMouseCursor(MMouseCursor cursor) {
    // Do some cursor magic
    switch (cursor) {
      case Mouse_Normal:
        m_Handle->setCursor(Qt::ArrowCursor);
        break;
      case Mouse_SizeAll:
        m_Handle->setCursor(Qt::SizeAllCursor);
        break;
      case Mouse_SizeEastWest:
        m_Handle->setCursor(Qt::SizeVerCursor);
        break;
      case Mouse_SizeNorthSouth:
        m_Handle->setCursor(Qt::SizeHorCursor);
        break;
      default:
        // Reset the cursor for this widget
        m_Handle->unsetCursor();
    }
  }

  QWidget* MComponent::getHandle() {
    return m_Handle;
  }

  void MComponent::setBackgroundColour(const MColour& colour) {
    bgColour = colour;

    // if we have an alpha of zero, then we have basically unset the colour, so indicate as such.
    if (colour.a <= 0.0) {
      bgColourSet = false;
    } else {
      bgColourSet = true;
    }

    if (m_Handle) {
/*      QColor col;

      col.setRgb(bgColour.r, bgColour.g, bgColour.b);*/
      //m_Handle->/*setPaletteBackgroundColor*/setColor(QColor(bgColour.r, bgColour.g, bgColour.b));
      /*QPalette pal(QColor(bgColour.r, bgColour.g, bgColour.b));
      m_Handle->setPalette(pal);*/
      //m_Handle->setEraseColor(QColor (0, 0, 0));
    }
    //refresh();
  }

  void MComponent::setForegroundColour(const MColour& colour) {
    fgColour = colour;

    if (colour.a <= 0.0) {
      fgColourSet = false;
    } else {
      fgColourSet = true;
    }

    if (m_Handle && fgColourSet) {
/*      QColor col;

      col.setRgb(fgColour.r, fgColour.g, fgColour.b);*/
      //m_Handle->/*setPaletteForegroundColor*/setColor(QColor(bgColour.r, bgColour.g, bgColour.b));
      /*QPalette pal(QColor(fgColour.r, fgColour.g, fgColour.b));
      m_Handle->setPalette(pal);*/
      //m_Handle->setEraseColor(QColor (0, 0, 0));
    }
    //refresh();
  }

  bool MComponent::onMouseMove(const MMouseEvent &event) {
    return false;
  }

  bool MComponent::onMousePressed(const MMouseEvent &event) {
    return false;
  }

  bool MComponent::onMouseReleased(const MMouseEvent &event) {
    return false;
  }

  bool MComponent::onMouseDoubleClick(const MMouseEvent &event) {
    return false;
  }

  bool MComponent::onMouseWheel(const MMouseEvent &event) {
    return false;
  }

  bool MComponent::onResize(int newWidth, int newHeight) {
    return false;
  }

  bool MComponent::onShow() {
    return true;
  }

  bool MComponent::onPaint() {
    return false;
  }

  void MComponent::onCreate() {
  }

  bool MComponent::onKeyPressed(const MKeyEvent &event) {
    return false;
  }

  bool MComponent::onKeyReleased(const MKeyEvent &event) {
    return false;
  }

  void MComponent::addMouseEventListener(const MMouseEventListenerPtr &listener) {
    mouseListeners.push_back(listener);
  }

  void MComponent::addKeyEventListener(const MKeyEventListenerPtr &listener) {
    keyListeners.push_back(listener);
  }

  void MComponent::fireMouseListeners(const MMouseEvent &event) {
    for (int i = 0; i < mouseListeners.size(); ++i) {
      mouseListeners[i]->onMouseEvent(this, event);
    }
  }

  void MComponent::fireKeyListeners(const MKeyEvent &event) {
    for (int i = 0; i < keyListeners.size(); ++i) {
      keyListeners[i]->onKeyEvent(this, event);
    }
  }

  bool MComponent::handleKeyEvent(const MKeyEvent &event) {
    bool handled = false;
    fireKeyListeners(event);
    handled = this->onKeyPressed(event);

    if (!handled) {
      handled = MApp::getInstance()->onKeyPressed(event);
    }

    return handled;
  }

}

