#include <AztecGUICommonPCH.h>
#include <gui/MBorderLayout.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

  MConstraint MBorderLayout::NORTH = MConstraint(1);
  MConstraint MBorderLayout::SOUTH = MConstraint(2);
  MConstraint MBorderLayout::EAST = MConstraint(3);
  MConstraint MBorderLayout::WEST = MConstraint(4);
  MConstraint MBorderLayout::CENTRE = MConstraint(5);
  MConstraint MBorderLayout::NORTH_INNER = MConstraint(6);
  MConstraint MBorderLayout::SOUTH_INNER = MConstraint(7);

  MBorderLayout::MBorderLayout(int horizBorder, int vertBorder, int horizGap, int vertGap)
    : m_HBorder(horizBorder), m_VBorder(vertBorder), m_HGap(horizGap), m_VGap(vertGap) 
  { 
  }

  MBorderLayout::~MBorderLayout() { 
  }

  // Layoutmanager methods
  void MBorderLayout::addComponent(MComponentPtr component) {
    addComponent(component, CENTRE);
  }

  void MBorderLayout::addComponent(MComponentPtr component, MConstraint constraints) {
    if (constraints == CENTRE) {
      m_Centre = component;
    } else if (constraints == NORTH) {
      m_North = component;
    } else if (constraints == SOUTH) {
      m_South = component;
    } else if (constraints == EAST) {
      m_East = component;
    } else if (constraints == WEST) {
      m_West = component;
    } else if (constraints == NORTH_INNER) {
      m_InnerNorth = component;
    } else if (constraints == SOUTH_INNER) {
      m_InnerSouth = component;
    }
  }

  void MBorderLayout::removeComponent(MComponentPtr component) {
    if (m_Centre == component) {
      m_Centre = NULL;
    } else if (m_North == component) {
      m_North = NULL;
    } else if (m_South == component) {
      m_South = NULL;
    } else if (m_East == component) {
      m_East = NULL;
    } else if (m_East == component) {
      m_East = NULL;
    } else if (m_West == component) {
      m_West = NULL;
    } else if (m_InnerNorth == component) {
      m_InnerNorth = NULL;
    } else if (m_InnerSouth == component) {
      m_InnerSouth = NULL;
    }
  }

  void MBorderLayout::layoutContainer(MContainerPtr container) {
    MRect2D rect;
    MSize2D size;

    rect = container->getClientSize();

    rect.x1 += m_HBorder;
    rect.x2 -= m_HBorder;
    rect.y1 += m_VBorder;
    rect.y2 -= m_VBorder;

    // put the north south ones down apppropiately.
    if (m_North != NULL) {
      size = m_North->getPreferredSize();
      m_North->setPosition(rect.x1, rect.y1, rect.getWidth(), size.getHeight());

      rect.y1 += m_North->getSize().getHeight() + m_VGap;
    }

    if (m_South != NULL) {
      size = m_South->getPreferredSize();
      m_South->setPosition(rect.x1, rect.y2 - size.getHeight(), rect.getWidth(), size.getHeight());

      rect.y2 -= m_South->getSize().getHeight() + m_VGap;
    }

    if (m_West != NULL) {
      size = m_West->getPreferredSize();
      m_West->setPosition(rect.x1, rect.y1, size.getWidth(), rect.getHeight());

      rect.x1 += m_West->getSize().getWidth() + m_HGap;
    }

    if (m_East != NULL) {
      size = m_East->getPreferredSize();
      m_East->setPosition(rect.x2 - size.getWidth(), rect.y1, size.getWidth(), rect.getHeight());

      rect.x2 -= m_East->getSize().getWidth() + m_HGap;
    }

    if (m_InnerNorth != NULL) {
      size = m_InnerNorth->getPreferredSize();
      m_InnerNorth->setPosition(rect.x1, rect.y1, rect.getWidth(), size.getHeight());

      rect.y1 += m_InnerNorth->getSize().getHeight() + m_VGap;
    }

    if (m_InnerSouth != NULL) {
      size = m_InnerSouth->getPreferredSize();
      m_InnerSouth->setPosition(rect.x1, rect.y2 - size.getHeight(), rect.getWidth(), size.getHeight());

      rect.y2 -= m_InnerSouth->getSize().getHeight() + m_VGap;
    }


    if (m_Centre != NULL) {
      m_Centre->setWindowRect(rect);
    }
  
  }


  /**
   * This gets the minimum size of this layout. 
   */
  MSize2D MBorderLayout::getMinimumSize(MContainerPtr container) {
    MSize2D size;

    if (m_Centre != NULL) {
      size = m_Centre->getMinimumSize();
    }

    doVerticalExtent(size, m_North, true);
    doVerticalExtent(size, m_South, true);
    doVerticalExtent(size, m_InnerNorth, true);
    doVerticalExtent(size, m_InnerSouth, true);


    doHorizExtent(size, m_East, true);
    doHorizExtent(size, m_West, true);

    size.adjustWidth(m_HGap);
    size.adjustHeight(m_VGap);

    return size;
  }

  MSize2D MBorderLayout::getPreferredSize(MContainerPtr container) {
    MSize2D size;

    if (m_Centre != NULL) {
      size = m_Centre->getPreferredSize();
    }

    doVerticalExtent(size, m_North);
    doVerticalExtent(size, m_South);
    doVerticalExtent(size, m_InnerNorth, true);
    doVerticalExtent(size, m_InnerSouth, true);

    doHorizExtent(size, m_East);
    doHorizExtent(size, m_West);


    size.adjustWidth(m_HGap);
    size.adjustHeight(m_VGap);

    return size;
  }

  void MBorderLayout::doHorizExtent(MSize2D &size, const MComponentPtr &comp, bool minimum ) {
    if (comp == NULL) {
      return;
    }

    MSize2D compSize;
    if (minimum) {
      compSize = comp->getMinimumSize();
    } else {
      compSize = comp->getPreferredSize();
    }

    size.adjustWidth(compSize.getWidth() + m_HGap);
    if (compSize.getHeight() + m_VGap > size.getHeight()) size.setHeight(compSize.getHeight() + m_VGap);
  }

  void MBorderLayout::doVerticalExtent(MSize2D &size, const MComponentPtr &comp, bool minimum ) {
    if (comp == NULL) {
      return;
    }

    MSize2D compSize;
    if (minimum) {
      compSize = comp->getMinimumSize();
    } else {
      compSize = comp->getPreferredSize();
    }
    if (compSize.getWidth() + m_HGap > size.getWidth()) size.setWidth(compSize.getWidth() + m_HGap);
    size.adjustHeight(compSize.getHeight() + m_VGap);
  }

  MConstraint MBorderLayout::getConstraint(const MComponentPtr &component) {
    // just run through all our known components and return the appropriate constraint.
    if (component == m_North) {
      return NORTH;
    } else if (component == m_South) {
      return SOUTH;
    } else if (component == m_East) {
      return EAST;
    } else if (component == m_West) {
      return WEST;
    } else if (component == m_InnerSouth) {
      return SOUTH_INNER;
    } else if (component == m_InnerNorth) {
      return NORTH_INNER;
    } else if (component == m_Centre) {
      return CENTRE;
    } else {
      // return an empty constraint if we couldn't find it.
      return MConstraint();
    }

  }


}
