#include <Aztec3DPCH.h>
#include <tools/MCreatePolyTool.h>


// Aztec2 includes
#include <views/Aztec3DView.h>
#include <utils/AztecGLUtils.h>
#include <config/UIConfig.h>

// AztecLib includes
#include <MUIManager.h>
#include <MScene.h>
#include <scripting/MScriptInterp.h>

namespace AztecGUI {
  
  MCreatePolyTool::MCreatePolyTool()
  {
    dragging = false;
    startingVertex = -1;
    startingTriangle = -1;
  }

  std::string MCreatePolyTool::getName() {
    return "toolCreatePoly";
  }

  int MCreatePolyTool::drawTool(bool select, const Aztec::MComponentPtr &component)
  {
    return 0;
  }

  void MCreatePolyTool::initialise() {
    startingVertex = -1;
    startingTriangle = -1;

    // start us off in point mode.
    Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.componentModeSet('point')");
  }

  bool MCreatePolyTool::finish() {
    // stop doing anything
    dragging = false;
    startingVertex = -1;
    startingTriangle = -1;

    Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.componentModeSet('object')");

    meshObj = NULL;

    // return true because we are finished with the tool.
    return true;
  }

  bool MCreatePolyTool::cancel() {
    // delete the vertices and triangles we made.
    getMeshObject();

    if (startingVertex > -1) {
      while (meshObj->getNumVerts() > startingVertex) {
        meshObj->deleteVertex(meshObj->getNumVerts() - 1);
      }
    }
    if (startingTriangle > -1) {
      while (meshObj->getNumTris() > startingTriangle) {
        meshObj->deleteVertex(meshObj->getNumTris() - 1);
      }
    }

    // stop doing anything
    // if we are triangulating a polygon, then we just want to cancel this one, so return false.
    // but if we aren't in the middle of something, return true to gfo back to the select tool.

    bool doneWithTool = true;

    if (startingVertex != -1 || startingTriangle != -1) {
      doneWithTool = false;
    }

    dragging = false;
    startingVertex = -1;
    startingTriangle = -1;

    if (doneWithTool) {
      meshObj = NULL;
    }

    return doneWithTool;
  }

  bool MCreatePolyTool::inProgress() {
    return (startingVertex != -1) || (startingTriangle != -1);
  };
  
  int MCreatePolyTool::onMouseDown(const Aztec::MMouseEvent &event)
  {
    MXYZToolType::onMouseDown(event);
    
    
    // check to see what component mode we are in. If we are not in point mode,
    // then do nothing
    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::POINT_TYPE) {
      
      AztecGLCanvasPtr glView = AztecGLView::getGLCanvasFor(event.getComponent());
      
      if (glView == NULL) {
        return TOOLRESULT_DRAWNONE;
      }
      
      
      Aztec::MVector3 newPoint = getPositionFromMouse(event);
      
      //Create a new point when left button is clicked
      if (event.getShiftState().leftMouseDown) {
        dragging = true;
        getMeshObject();
        
        // deselect all the previous points.
        for (int index = 0; index < meshObj->getComponentCount(Aztec::MComponentisedObject::POINT_TYPE); ++index) {
          meshObj->flagComponent(Aztec::MComponentisedObject::POINT_TYPE, 
            index, 
            Aztec::MComponentisedObject::COMPONENT_SELECTED,
            Aztec::MComponentisedObject::atUnset);
        }

        // if we don't have a starting vertex, then add one in.
        if (startingVertex == -1) {
          startingVertex = meshObj->getNumVerts();
        }

        meshObj->addVertex(newPoint.x,newPoint.y,newPoint.z);
        meshObj->flagComponent(
          Aztec::MComponentisedObject::POINT_TYPE, 
          meshObj->getNumVerts()-1,
          Aztec::MComponentisedObject::COMPONENT_SELECTED
          );
        
        // if we have more than two verts, we can join them together as a face

        if(meshObj->getNumVerts() > 2) {
          // if we don't have a starting triangle, now is the time to start one.
          if (startingTriangle == -1) {
            startingTriangle = meshObj->getNumTris();;
          }
        
          while (meshObj->getNumTris() > startingTriangle) {
            meshObj->deleteTriangle(meshObj->getNumTris() - 1);
          }
          
          std::vector<int> points;
          for (int i = startingVertex; i < meshObj->getNumVerts(); ++i) {
            points.push_back(i);
          }
          
          meshObj->triangulatePolygon(points);
        }
        
        return TOOLRESULT_DRAWALL;
        
      } else if (event.getShiftState().middleMouseDown) {
        dragging = true;
        getMeshObject();
        
        meshObj->setVertexPosition(meshObj->getNumVerts() - 1, newPoint);
        return TOOLRESULT_DRAWALL;
      }
      
    }
    return TOOLRESULT_DRAWNONE;
  }
  
  int MCreatePolyTool::onMouseMove(const Aztec::MMouseEvent &event) {
    if (dragging) {
      // if we are dragging it means we are adjusting the newly added point
      // so lets move it around.

      getMeshObject();

      Aztec::MVector3 newPoint = getPositionFromMouse(event);

      meshObj->setVertexPosition(meshObj->getNumVerts() - 1, newPoint);
      return TOOLRESULT_DRAWALL;
    }

    return TOOLRESULT_DRAWNONE;
  }

  int MCreatePolyTool::onMouseUp(const Aztec::MMouseEvent &event) {
    dragging = false;
    return TOOLRESULT_DRAWNONE;
  }

  void MCreatePolyTool::getMeshObject() {
    // if we already have a mesh object, then we don't need to go looking
    if (meshObj != NULL) {
      return;
    }

    shapeObj = NULL;
    meshObj = NULL;
    sceneObj = NULL;

    // if we have no mesh shape at all, we must make one.
    if (shapeObj == NULL) {
      sceneObj = new Aztec::MSceneObject();
      shapeObj = new Aztec::MMeshShape();
      meshObj = new Aztec::MEditableMesh();

      shapeObj->setMeshObject(meshObj);
      sceneObj->setShapeObject(shapeObj);

      sceneObj->setName("Mesh");
      shapeObj->setName("MeshShape");
      meshObj->setName("MeshMesh");

      Aztec::MScene::getGlobalScene()->addObject(sceneObj);
      Aztec::MScene::getGlobalScene()->addObject(shapeObj);

      Aztec::MScene::getGlobalScene()->selectObject(sceneObj);

    } else {
		meshObj = AZTEC_CAST(Aztec::MEditableMesh, shapeObj->getMeshObject());

		/*std::vector<int> pointsToUse;
		pointsToUse.push_back(0);
		pointsToUse.push_back(1);
		meshObj->triangulatePolygon(pointsToUse);*/

    }

    // change to point mode so we can see the points we are making
    Aztec::MUIManager::setComponentMode(Aztec::MComponentisedObject::POINT_TYPE);
  }


  Aztec::MVector3 MCreatePolyTool::getPositionFromMouse(const Aztec::MMouseEvent &event) {
    AztecGLCanvasPtr glView = AztecGLView::getGLCanvasFor(event.getComponent());

    // add the new point in.
    Aztec::MRay ray = glView->getRay(event.getX(), event.getY());
    Aztec::MPlane plane = glView->getGridPlane();

    return ray.intersectWithPlane(plane);
  }

  int MCreatePolyTool::getDefaultManip() {
    return -1;
  }
}