#include <Aztec3DPCH.h>

// Aztec2 includes
#include <functions/tools/ToolFunctions.h>
#include <tools/MToolManager.h>
#include <views/AztecViewManager.h>

// AztecGUICommon
#include <config/UIConfig.h>
#include <MUIManager.h>

// AztecLib includes
#include <MSystemManager.h>

namespace AztecGUI {

  /**
   * This sets the current tool that is currenty in use for a group of views.
   *
   * @args Must contain two arguments:
   *          arg0: The view group that we are dealing with. e.g "3D", or "graphView"
   *          arg1: the name of the tool to set to.
   */
  static int toolSet(const StringVector &args, std::string &result) {
    // Make sure we have the right number of args
    if (args.size() != 2) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: toolSetCurrent() - requires two arguments. e.g: scene.toolSetCurrent(\"3DGroup\", \"MSelectTool\")");
      return FunctionManager::FAIL;
    }

    // get the args out of the argument array.
    const std::string &viewGroup = args[0];
    const std::string &toolName = args[1];

    MToolTypePtr tool = MToolManager::getInstance()->getToolByName(toolName);

    // If we got a valid tool, set it as the current one, and off we go, otherwise report an error.
    if (tool != NULL) {
      MToolManager::getInstance()->setTool(tool, viewGroup);

      // update the screen
      AztecViewManager::redrawAllViews();

      return FunctionManager::SUCCEED;
    } else {
      Aztec::MSystemManager::getInstance()->logOutput("Error: toolSetCurrent() - could find tool '%s'", toolName.c_str());
      return FunctionManager::FAIL;
    }
  }

  static int toolCancel(const StringVector &args, std::string &result) {
    AztecViewPtr currentView = AztecViewManager::getCurrentView();

    // if we do not have a current view, then we can't cancel the current tool.
    if (currentView == NULL) {
      return FunctionManager::FAIL;
    }

    // Get the current tool for that view
    MToolTypePtr tool = MToolManager::getInstance()->GetTool(currentView->getViewGroup());

    // if we don't have a tool, then we can't cancel it.
    if (tool == NULL) {
      MToolManager::getInstance()->setTool(currentView->getSelectTool(), currentView->getViewGroup());
      return FunctionManager::FAIL;
    }

    // If the cancel method returns true, it means we are finished with that 
    // tool, and should go back to our previous tool.
    if (tool->cancel()) {
      MToolManager::getInstance()->PopTool(currentView->getViewGroup());
      AztecViewManager::redrawAllViews();
    }

    return FunctionManager::SUCCEED;
  }

  static int toolFinish(const StringVector &args, std::string &result) {
    AztecViewPtr currentView = AztecViewManager::getCurrentView();

    // if we do not have a current view, then we can't finish the current tool.
    if (currentView == NULL) {
      return FunctionManager::FAIL;
    }

    // Get the current tool for that view
    MToolTypePtr tool = MToolManager::getInstance()->GetTool(currentView->getViewGroup());

    // if we don't have a tool, then we can't finish it.
    if (tool == NULL) {
      MToolManager::getInstance()->setTool(currentView->getSelectTool(), currentView->getViewGroup());
      return FunctionManager::FAIL;
    }

    // If the finish method returns true, it means we are finished with that 
    // tool, and should go back to our previous tool.
    if (tool->finish()) {
      MToolManager::getInstance()->PopTool(currentView->getViewGroup());
      AztecViewManager::redrawAllViews();
    }

    return FunctionManager::SUCCEED;
  }

  static int toolMakeBigger(const StringVector &args, std::string &result) {
    UIConfig::set3DWidgetSize(UIConfig::get3DWidgetSize() * 1.5);
    return FunctionManager::SUCCEED;
  }

  static int toolMakeSmaller(const StringVector &args, std::string &result) {
    UIConfig::set3DWidgetSize(UIConfig::get3DWidgetSize() / 1.5);
    return FunctionManager::SUCCEED;
  }

  static int getSnapMode(const StringVector &args, std::string &result) {
	  int mode = Aztec::MUIManager::getSnapMode();
	  result = Aztec::intToStr(mode).c_str();
    return FunctionManager::SUCCEED;
  }

  static int setSnapGrid(const StringVector &args, std::string &result) {
	Aztec::MUIManager::setSnapMode(Aztec::MUIManager::SNAP_GRID);
    return FunctionManager::SUCCEED;
  }
  static int setSnapPoint(const StringVector &args, std::string &result) {
	Aztec::MUIManager::setSnapMode(Aztec::MUIManager::SNAP_POINT);
    return FunctionManager::SUCCEED;
  }

  static int setSnapNone(const StringVector &args, std::string &result) {
	Aztec::MUIManager::setSnapMode(Aztec::MUIManager::SNAP_NONE);
    return FunctionManager::SUCCEED;
  }




  void registerToolFunctions(FunctionManager &man) {
    man.registerFunction("toolSetCurrent", toolSet);
    man.registerFunction("toolCancel", toolCancel);
    man.registerFunction("toolFinish", toolFinish);
    man.registerFunction("toolMakeBigger", toolMakeBigger);
    man.registerFunction("toolMakeSmaller", toolMakeSmaller);
	man.registerFunction("getSnapMode", getSnapMode);
	man.registerFunction("setSnapGrid", setSnapGrid);
	man.registerFunction("setSnapPoint", setSnapPoint);
	man.registerFunction("setSnapNone", setSnapNone);

  }

}

