#include <Aztec3DPCH.h>

// Aztec2 includes
#include <functions/FunctionManager.h>
#include <views/AztecViewManager.h>
#include <utils/SceneFunctions.h>
#include <functions/mesh/MeshFunctions.h>

// AztecLib includes
#include <MSystemManager.h>
#include <MUIManager.h>
#include <MEditableMesh.h>
#include <misc/MSceneHelper.h>

// Standard includes
#include <algorithm>
#include <assert.h>
#include <math.h>

namespace AztecGUI {

  using namespace Aztec;

  typedef std::set<int> TriangleSet;
  typedef std::vector<TriangleSet> TriangleSetVector;

  typedef std::set<int> IntSet;
  typedef std::map<int, int> IntMap;
  typedef std::vector<IntMap> IntMapVector;


  void getSelectedGroupFrom(const MEditableMeshPtr &mesh, AztecFlags oldflag, AztecFlags newflag, int triangle, TriangleSet &tris, IntMap &allEdges) {
    mesh->unsetTriangleFlag(triangle, oldflag);
    mesh->setTriangleFlag(triangle, newflag);
    tris.insert(triangle);

    for (int edge = 0; edge < 3; ++edge) {
      int neighbour = mesh->getTriangleNeighbour(triangle, edge);

      if (neighbour != -1 && mesh->isTriangleFlagged(neighbour, oldflag)) {
        getSelectedGroupFrom(mesh, oldflag, newflag, neighbour, tris, allEdges);
      }
      if (mesh->isTriangleFlagged(triangle, TRIANGLE_SELECTED) && !mesh->isTriangleFlagged(neighbour, TRIANGLE_SELECTED)) {
        int a, b;
        mesh->getVertsFromEdge(triangle, edge, &a, &b);
        allEdges[a] = b;
      }
    }



  }

  inline float vertexDistance(const std::map<int, MVector3> &mesh, int from, int to) {
    std::map<int, MVector3>::const_iterator itFrom = mesh.find(from);
    std::map<int, MVector3>::const_iterator itTo = mesh.find(to);

    assert(itFrom != mesh.end());
    assert(itTo != mesh.end());
    return (itFrom->second - itTo->second).length2();
  }

  inline float vertexDistance(const MEditableMeshPtr &mesh, int from, int to) {
    return (mesh->getVertexPosition(from) - mesh->getVertexPosition(to)).length2();
  }

  void getSelectedFaceGroups(const MEditableMeshPtr &mesh, TriangleSetVector &triSets, IntMapVector &allEdges) {

    triSets.clear();

    for (int tri = 0; tri < mesh->getNumTris(); ++tri) {
      if (mesh->isTriangleFlagged(tri, TRIANGLE_SELECTED)) {
        mesh->setTriangleFlag(tri, TRIANGLE_FLAGFORCHANGE);
      }
    }


    // now find our seed triangle that has the TRIANGLE_FLAGFORCHANGE on it
    for (int tri = 0; tri < mesh->getNumTris(); ++tri) {
      if (mesh->isTriangleFlagged(tri, TRIANGLE_FLAGFORCHANGE)) {
        triSets.push_back(TriangleSet());
        allEdges.push_back(IntMap());
        getSelectedGroupFrom(mesh, TRIANGLE_FLAGFORCHANGE, TRIANGLE_FLAGFORCHANGE2, tri, triSets.back(), allEdges.back());
      }

    }
    
  }

  MVector3 getCentre(const MEditableMeshPtr &mesh, const IntMap &poly) {
    if (poly.size() == 0) return MVector3();
    MVector3 total;
    for (IntMap::const_iterator it = poly.begin(); it != poly.end(); ++it) {
      total += mesh->getVertexPosition(it->first);
    }

    return total / (float)poly.size();
  }

  MVector3 getNormal(const MEditableMeshPtr &mesh, const IntMap &poly, const MVector3 &centre) {
    if (poly.size() == 0) return MVector3();

    MVector3 normal;
    for (IntMap::const_iterator it = poly.begin(); it != poly.end(); ++it) {
      MVector3 a = mesh->getVertexPosition(it->first) - centre;
      MVector3 b = mesh->getVertexPosition(it->second) - centre;

      normal += a / b;
    }

    normal.normalize();
    return normal;
  }

#ifdef _DEBUG 

}

#include <MSceneObject.h>
#include <MLineShape.h>
#include <MLineMesh.h>

namespace AztecGUI {
  using namespace Aztec;

  void createLineShapeFrom(const IntMap &points, const std::map<int, MVector3> &positions) {
    MSceneObjectPtr sceneObj = new Aztec::MSceneObject();
    MLineShapePtr shapeObj = new Aztec::MLineShape();
    MLineMeshPtr meshObj = new Aztec::MLineMesh();
    
    shapeObj->setLineMesh(meshObj);
    sceneObj->setShapeObject(shapeObj);
    
    sceneObj->setName("line");
    shapeObj->setName("lineShape");
    meshObj->setName("lineMesh");
    
    Aztec::MScene::getGlobalScene()->addObject(sceneObj);
    Aztec::MScene::getGlobalScene()->addObject(shapeObj);
    
    int start = points.begin()->first;
    int prev = -1;
    int current = start;

    int count = 0;
    do {
      meshObj->addVertex(positions.find(current)->second);

      if (prev != -1) {
        meshObj->addEdge(meshObj->getVertexCount()-2, meshObj->getVertexCount()-1); 
      }

      prev = current;
      current = points.find(current)->second;
    } while (current != start);

    meshObj->addEdge(meshObj->getVertexCount()-1, 0); 
  }

#endif

  void makeInsideUnitSquare(const IntMap &poly, std::map<int, MVector3> &vertices) {
    MVector3 min = vertices[poly.begin()->first];
    MVector3 max = vertices[poly.begin()->first];
    for (IntMap::const_iterator it = poly.begin(); it != poly.end(); ++it) {
      const MVector3 &p = vertices[it->first];
      if (p.x < min.x) min.x = p.x;
      if (p.y < min.y) min.y = p.y;
      if (p.z < min.z) min.z = p.z;
      if (p.x > max.x) max.x = p.x;
      if (p.y > max.y) max.y = p.y;
      if (p.z > max.z) max.z = p.z;
    }
    MVector3 diff = max - min;
    for (IntMap::const_iterator it = poly.begin(); it != poly.end(); ++it) {
      MVector3 &p = vertices[it->first];
      p -= min;
      if (diff.x > 0.001) {
        p.x /= diff.x;
      }
      if (diff.y > 0.001) {
        p.y /= diff.y;
      }
    }

  }

  void triangulateBetween(const MEditableMeshPtr &mesh, const IntMap &from, const IntMap &to) {

    // find the two closest points on each polygon
    IntMap poly1 = from;

    MVector3 fromCentre = getCentre(mesh, from);
    MVector3 toCentre = getCentre(mesh, to);
    MVector3 fromNormal = getNormal(mesh, from, fromCentre);
    MVector3 toNormal = getNormal(mesh, to, fromCentre);

    // if the polygons are facing different directions, one of them must be 
    // reverse in order, so fix up the ordering.
    IntMap poly2;
    if (fromNormal * toNormal < 0) {
      for (IntMap::const_iterator it = to.begin(); it != to.end(); ++it) {
        poly2[it->second] = it->first;
      }
      // flip the direction of the normal.
      toNormal *= -1.0;
    } else {
      poly2 = to;
    }


    MVector3 averageCentre = 0.5 * (fromCentre + toCentre);
    MVector3 averageNormal = fromNormal + toNormal;
    averageNormal.normalize();

    std::map<int, MVector3> transformedVertices;

    for (IntMap::iterator it1 = poly1.begin(); it1 != poly1.end(); ++it1) {
      MVector3 pos = mesh->getVertexPosition(it1->first);
      pos -= fromCentre;
      pos -= averageNormal * (pos * averageNormal);
      transformedVertices[it1->first] = pos;
    }
    for (IntMap::iterator it2 = poly2.begin(); it2 != poly2.end(); ++it2) {
      MVector3 pos = mesh->getVertexPosition(it2->first);
      pos -= toCentre;
      pos -= averageNormal * (pos * averageNormal);
      transformedVertices[it2->first] = pos;
    }

    MVector3 basisX, basisY;

    {
      int first = poly1.begin()->first;
      int second = poly1.begin()->second;
      MVector3 n = averageNormal;
      MVector3 a = transformedVertices[first] - transformedVertices[second];
      a.normalize();

      basisX = n / a;
      basisY = basisX / n;
    }

    // go through the transformed coordinates, and place them on the xy place.
    for (std::map<int, MVector3>::iterator it = transformedVertices.begin(); it != transformedVertices.end(); ++it) {
      float u = it->second * basisX;
      float v = it->second * basisY;

      it->second.set(10*u,10*v,0);
    }

    makeInsideUnitSquare(poly1, transformedVertices);
    makeInsideUnitSquare(poly2, transformedVertices);


#ifdef _DEBUG
    createLineShapeFrom(poly1, transformedVertices);
    createLineShapeFrom(poly2, transformedVertices);
#endif


    int startA = -1, startB = -1;
    float distance2 = 0.0;

    for (IntMap::iterator it1 = poly1.begin(); it1 != poly1.end(); ++it1) {
      for (IntMap::iterator it2 = poly2.begin(); it2 != poly2.end(); ++it2) {
        if (startA == -1 || startB == -1) {
          startA = it1->second;
          startB = it2->second;
          distance2 = vertexDistance(transformedVertices, startA, startB);
        } else {
          float newDistance = vertexDistance(transformedVertices, it1->second, it2->second);
          if (newDistance < distance2) {
            startA = it1->second;
            startB = it2->second;
            distance2 = newDistance;
          }
        }
      }
    }

    int a = startA;
    int b = startB;

    do {
      // walk along the poly, we find out which vertex along each polygon will have the shortest edge length.
      int nextA = poly1[a];
      int nextB = poly2[b];

      float dist_A_nextB = nextB != -1 ? vertexDistance(transformedVertices, a, nextB) : 0.0;
      float dist_nextA_B = nextA != -1 ? vertexDistance(transformedVertices, nextA, b) : 0.0;

      if (nextB != -1 && ((dist_A_nextB < dist_nextA_B) || nextA == -1)) {
        mesh->addTriangle(b, a, nextB);
        mesh->setTriangleFlag(mesh->getNumTris() - 1, TRIANGLE_SELECTED);
        poly2[b] = -1;
        b = nextB;
      } else if (nextA != -1) {
        mesh->addTriangle(b, a, nextA);
        mesh->setTriangleFlag(mesh->getNumTris() - 1, TRIANGLE_SELECTED);
        poly1[a] = -1;
        a = nextA;
      } else {
        // semething really bad has happened that has caused us to end up 
        // with two non closed polygon loops.
        assert(0);
      }

    } while (a != startA || b != startB);

    // select the edges as a test.
    for (IntMap::iterator it = poly1.begin(); it != poly1.end(); ++it) {
      mesh->setEdgeFlag(it->first, it->second, EDGE_SELECTED);
    }
    for (IntMap::iterator it = poly2.begin(); it != poly2.end(); ++it) {
      mesh->setEdgeFlag(it->first, it->second, EDGE_SELECTED);
    }
    
  }

  /**
   * This function takes a mesh who has two separate selections of faces, and 
   * joins those selections together by a polygonal tube.
   */
  int faceConnectMesh(const MEditableMeshPtr &mesh) {
    // The first step is to extract the selected faces, and the edges that border that selection.
    TriangleSetVector triSets;
    IntMapVector allEdges;

    getSelectedFaceGroups(mesh, triSets, allEdges);

    // ensure that we only have two groups to connect
    if (triSets.size() != 2) {
      MSystemManager::getInstance()->logOutput("Warning: faceConnectMesh() failed. We need exactly two distinct sets of faces selected");
      return FunctionManager::FAIL;
    }

    // Delete the old triangles.
    mesh->deleteTriangleFlag(TRIANGLE_SELECTED);

    // make the new triangles
    triangulateBetween(mesh, allEdges[0], allEdges[1]);

    return FunctionManager::SUCCEED;
  }

  void doFaceConnect(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    MNamedObjectPtr namedObj = AZTEC_CAST(MNamedObject, node->getObject());
    MEditableMeshPtr mesh = getEditableMeshObject(namedObj);
    if (mesh == NULL) {
      MSystemManager::getInstance()->logOutput("Warning: edgeConnect() failed on object '%s', only operates on editable meshes", namedObj != NULL ? namedObj->getName().c_str() : "");
      return;
    }

    if (mesh->getComponentMode() == MComponentisedObject::EDGE_TYPE) {
      faceConnectMesh(mesh);
    }
  }

  int faceConnect(const StringVector &args, std::string &result) {
    applyToObjects(MScene::getGlobalScene(), MScene::getGlobalScene()->getObjectList(), ifSelectedCriteria, doFaceConnect);

    AztecViewManager::redrawAllViews();
	  
    return FunctionManager::SUCCEED;
  }


}

