#include <Aztec3DPCH.h>

#include <functions/edit/ObjectFunctions.h>

// Aztec2 includes
#include <views/AztecViewManager.h>
#include <views/DeletingView.h>

// AztecGUI inclides
#include <config/UIConfig.h>

// AztecLib includes
#include <MScene.h>
#include <MSystemManager.h>
#include <MUIManager.h>
#include <MListsTrees.h>
#include <MEditableMesh.h>
#include <MUndoManager.h>
#include <scripting/MScriptInterp.h>

// standard includes
#include <algorithm>
#include <locale>

namespace AztecGUI {

  static int editObjectDelete(const StringVector &args, std::string &result) {
    Aztec::MBaseObjectListPtr ObjList;
    Aztec::MBaseObjectPtr Obj;
    bool DeleteChildren = true;

    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
    
      ObjList = Aztec::MScene::getGlobalScene()->getSelectedObjectList();
    
      /*
      Aztec::MParameterObjectPtr Var = g_ProgSet.GetToolParameter("KObjectDeleteSelected", "DeleteChildren");
      if (Var != NULL) {
        int val;
        if (Var->getValueInteger(val)) {
          DeleteChildren = (val != 0);
        }
        Var = NULL;
      }*/
      DeleteChildren = true;
    
      ObjList->beginIteration();
      while ((Obj = ObjList->getNext()) != NULL) {
        Aztec::MScene::getGlobalScene()->deleteObject(AZTEC_CAST(Aztec::MNamedObject, Obj), DeleteChildren);
      }
      ObjList->endIteration();

      AztecViewManager::redrawAllViews();

    } else {
      // Delete some vertices
    
      // go through the selection and see what objects can have vertices deleted
      Aztec::MEditableMeshPtr MeshObj;
      Aztec::MSceneObjectPtr SceneObj;
    
      ObjList = Aztec::MScene::getGlobalScene()->getSelectedObjectList();
    
      ObjList->beginIteration();
   
      while ((Obj = ObjList->getNext()) != NULL) {
        SceneObj = AZTEC_CAST(Aztec::MSceneObject, Obj);
      
        if (SceneObj == NULL || SceneObj->getShapeObject() == NULL) continue;
      
        MeshObj = AZTEC_CAST(Aztec::MEditableMesh, SceneObj->getShapeObject()->convertToMesh());
    
        if (MeshObj == NULL) continue;
    
        if (MeshObj->isInComponentMode(Aztec::MComponentisedObject::POINT_TYPE)) {
          MeshObj->deleteVertexFlag(VERTEX_SELECTED);
        } else if (MeshObj->isInComponentMode(Aztec::MComponentisedObject::FACET_TYPE)) {
          MeshObj->deleteTriangleFlag(TRIANGLE_SELECTED);
        } else {
          Aztec::MSystemManager::getInstance()->logOutput("Warning: Nothing Deleted. Cannot Delete Component");
        }
      
      }
      ObjList->endIteration();

      AztecViewManager::redrawAllViews();
    
    }

    return FunctionManager::SUCCEED;

  }


  static int editDelete(const StringVector &args, std::string &result) {
    DeletingView *deleter = AZTEC_CAST(DeletingView, AztecViewManager::getCurrentView());
    
    // try the viewport specific action
    if (deleter != NULL) {
      deleter->deleteSelected();
      AztecViewManager::redrawCurrentView();
      return FunctionManager::SUCCEED;
    } else {
      // otherwise we have failed, so do nothing.
      return FunctionManager::FAIL;
    }

  }

  static void updateObjectModes() {
    // change all objects currently in component mode back into object mode
    Aztec::MBaseObjectPtr Obj;
    Aztec::MMeshPtr mesh;
    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();

    scene->getObjectList()->beginIteration();
    
    while ((Obj = scene->getObjectList()->getNext()) != NULL) {
      if (!Obj->isFlagged(OBJECTFLAG_SELECTED)) {
        continue;
      }
      
      Aztec::MSceneObjectPtr sceneObj = AZTEC_CAST(Aztec::MSceneObject, Obj);
      if (sceneObj == NULL) {
        continue;
      }
      
      Aztec::MComponentisedObjectPtr compObj = sceneObj->getComponentObject();
      
      if (compObj != NULL) {
        compObj->setComponentMode(Aztec::MUIManager::getComponentMode());
      }
    }
    scene->getObjectList()->endIteration();  
  }

  static int componentModeGet(const StringVector &args, std::string &result) {
    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::POINT_TYPE) {
      result = "point";
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::FACET_TYPE) {
      result = "facet";
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::EDGE_TYPE) {
      result = "edge";
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
      result = "object";
    } else {
      return FunctionManager::FAIL;
    }

    return FunctionManager::SUCCEED;
  }

  static int componentModeSet(const StringVector &args, std::string &result) {
    if (args.size() != 1) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: componentModeSet() - requires one arguments. e.g: scene.componentModeSet('point')");
      return FunctionManager::FAIL;
    }

    std::string current;
    componentModeGet(StringVector(), current);

    if (args[0] != current) {
      if (args[0] == "point") {
        Aztec::MUIManager::setComponentMode(Aztec::MComponentisedObject::POINT_TYPE);
      } else if (args[0] == "facet") {
        Aztec::MUIManager::setComponentMode(Aztec::MComponentisedObject::FACET_TYPE);
      } else if (args[0] == "edge") {
        Aztec::MUIManager::setComponentMode(Aztec::MComponentisedObject::EDGE_TYPE);
      } else if (args[0] == "object") {
        Aztec::MUIManager::setComponentMode(Aztec::MComponentisedObject::OBJECT_TYPE);
      } 

      Aztec::MScriptInterpreter::getInstance()->ExecuteScript(
        "Scene.toolSetCurrent('3D', 'toolSelect')");
      AztecViewPtr view = AztecViewManager::getCurrentView();
      if (view != NULL) {
        Aztec::MScriptInterpreter::getInstance()->ExecuteScript(
          (std::string("Scene.toolSetCurrent('") + view->getViewGroup() + "', 'toolSelect')").c_str());
      }
      updateObjectModes();
      AztecViewManager::redrawCurrentView();
    }

    return FunctionManager::SUCCEED;
  }

  static int componentModeCycle(const StringVector &args, std::string &result) {
    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
      return componentModeSet(StringVector(1, "point"), result);
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::POINT_TYPE) {
      return componentModeSet(StringVector(1, "facet"), result);
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::FACET_TYPE) {
      return componentModeSet(StringVector(1, "edge"), result);
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::EDGE_TYPE) {
      return componentModeSet(StringVector(1, "object"), result);
    } else {
      return FunctionManager::FAIL;
    }
  }

  static int axisModeGet(const StringVector &args, std::string &result) {
    switch (Aztec::MUIManager::getAxisMode()) {
    case Aztec::MUIManager::WORLD_AXIS:
      result = "world"; return FunctionManager::SUCCEED;
    case Aztec::MUIManager::LOCAL_AXIS:
      result = "local"; return FunctionManager::SUCCEED;
    case Aztec::MUIManager::PARENT_AXIS:
      result = "parent"; return FunctionManager::SUCCEED;
    case Aztec::MUIManager::SCREEN_AXIS:
      result = "screen"; return FunctionManager::SUCCEED;
    case Aztec::MUIManager::OBJECT_AXIS:
      result = "object"; return FunctionManager::SUCCEED;
    default:
      return FunctionManager::FAIL;
    }

    return FunctionManager::SUCCEED;
  }


  static int axisModeSet(const StringVector &args, std::string &result) {
    if (args.size() != 1) {
      Aztec::MSystemManager::getInstance()->logOutput(
        "Error: axisModeSet requires one argument. Usage: axisModeSet(['world' | 'local' | 'parent' | 'screen' | 'object']);");
      return FunctionManager::FAIL;
    }

    std::string mode = args[0];

    std::transform(mode.begin(), mode.end(), mode.begin(), tolower);

    if (mode == "world") {
      Aztec::MUIManager::setAxisMode(Aztec::MUIManager::WORLD_AXIS);
    } else if (mode == "local") {
      Aztec::MUIManager::setAxisMode(Aztec::MUIManager::LOCAL_AXIS);
    } else if (mode == "parent") {
      Aztec::MUIManager::setAxisMode(Aztec::MUIManager::PARENT_AXIS);
    } else if (mode == "screen") {
      Aztec::MUIManager::setAxisMode(Aztec::MUIManager::SCREEN_AXIS);
    } else if (mode == "object") {
      Aztec::MUIManager::setAxisMode(Aztec::MUIManager::OBJECT_AXIS);
    } else {
      Aztec::MSystemManager::getInstance()->logOutput(
        "Error: axisModeSet didn't recoginise the mode '%s'", mode.c_str());
      return FunctionManager::FAIL;
    }

    return FunctionManager::SUCCEED;
  }

  static int axisModeCycle(const StringVector &args, std::string &result) {
    switch (Aztec::MUIManager::getAxisMode()) {
    case Aztec::MUIManager::WORLD_AXIS:
      Aztec::MUIManager::setAxisMode(Aztec::MUIManager::LOCAL_AXIS); return FunctionManager::SUCCEED;
    case Aztec::MUIManager::LOCAL_AXIS:
      Aztec::MUIManager::setAxisMode(Aztec::MUIManager::PARENT_AXIS); return FunctionManager::SUCCEED;
    case Aztec::MUIManager::PARENT_AXIS:
      Aztec::MUIManager::setAxisMode(Aztec::MUIManager::SCREEN_AXIS); return FunctionManager::SUCCEED;
    case Aztec::MUIManager::SCREEN_AXIS:
      Aztec::MUIManager::setAxisMode(Aztec::MUIManager::OBJECT_AXIS); return FunctionManager::SUCCEED;
    case Aztec::MUIManager::OBJECT_AXIS:
      Aztec::MUIManager::setAxisMode(Aztec::MUIManager::WORLD_AXIS); return FunctionManager::SUCCEED;
    default:
      return FunctionManager::FAIL;
    }

  }

  static int axisSeparateToggle(const StringVector &args, std::string &result) {
    Aztec::MUIManager::setSeparateAxisMode( !Aztec::MUIManager::getSeparateAxisMode() );
    return FunctionManager::SUCCEED;
  }

  static int undo(const StringVector &args, std::string &result) {
    Aztec::MUndoManager::getInstance()->endUndo();
    Aztec::MUndoManager::getInstance()->undo();
    Aztec::MUndoManager::getInstance()->beginUndo("");
    return FunctionManager::SUCCEED;
  }

  static int redo(const StringVector &args, std::string &result) {
    Aztec::MUndoManager::getInstance()->endUndo();
    Aztec::MUndoManager::getInstance()->redo();
    Aztec::MUndoManager::getInstance()->beginUndo("");
    return FunctionManager::SUCCEED;
  }

  static const char * EDIT_NAME = "Edit";
  void registerObjectFunctions(FunctionManager &man) {

    man.registerFunction("editDelete", editDelete, EDIT_NAME, "Delete selected");
    man.registerFunction("editObjectDelete", editObjectDelete, EDIT_NAME, "");

    man.registerFunction("componentModeSet", componentModeSet);
    man.registerFunction("componentModeGet", componentModeGet);
    man.registerFunction("componentModeCycle", componentModeCycle, EDIT_NAME, "Cycle Component Mode");

    AztecGUI::UIConfig::addFriendlyCommand("Object Mode", "Scene.componentModeSet('object')", EDIT_NAME);
    AztecGUI::UIConfig::addFriendlyCommand("Point Mode", "Scene.componentModeSet('point')", EDIT_NAME);
    AztecGUI::UIConfig::addFriendlyCommand("Face Mode", "Scene.componentModeSet('facet')", EDIT_NAME);
    AztecGUI::UIConfig::addFriendlyCommand("Edge Mode", "Scene.componentModeSet('edge')", EDIT_NAME);

    man.registerFunction("axisModeSet", axisModeSet);
    man.registerFunction("axisModeGet", axisModeGet);
    man.registerFunction("axisModeCycle", axisModeCycle, EDIT_NAME, "Cycle Axis Mode");
    man.registerFunction("axisSeparateToggle", axisSeparateToggle, EDIT_NAME, "Toggle Separate Axis Mode");

    AztecGUI::UIConfig::addFriendlyCommand("World Axis Mode", "Scene.axisModeSet('world')", EDIT_NAME);
    AztecGUI::UIConfig::addFriendlyCommand("Local Axis Mode", "Scene.axisModeSet('local')", EDIT_NAME);
    AztecGUI::UIConfig::addFriendlyCommand("Parent Axis Mode", "Scene.axisModeSet('parent')", EDIT_NAME);
    AztecGUI::UIConfig::addFriendlyCommand("Screen Axis Mode", "Scene.axisModeSet('screen')", EDIT_NAME);
    AztecGUI::UIConfig::addFriendlyCommand("Object Axis Mode", "Scene.axisModeSet('object')", EDIT_NAME);

    man.registerFunction("undo", undo, "Edit", "Undo");
    man.registerFunction("redo", redo, "Edit", "Redo");

    man.registerFunction("duplicate", duplicate, "Edit", "Duplicate Selected");
  }


}

