
#include <Aztec3DPCH.h>
#include <controls/TimeSlider.h>
#include <views/AztecViewManager.h>

#include <MUIManager.h>
#include <MScene.h>
#include <MKey.h>
#include <MSystemManager.h>

#include <assert.h>

#ifndef _WIN32
#include <qpainter.h>
#include <iostream>
#endif


namespace AztecGUI {

  static MTimeSlider *g_TimeSlider = NULL;

  MTimeSlider* MTimeSlider::getInstance() {
    return g_TimeSlider;
  }


  MTimeSlider::MTimeSlider() {
    assert(g_TimeSlider == NULL);
    g_TimeSlider = this;

    m_Tracking = false;
    m_ScrollPos = 0;
    m_ScrollMin = 0;
    m_ScrollMax = 100;

    setMinimumSize(Aztec::MSize2D(32,32));
  }

  bool MTimeSlider::setScrollRange(int Min, int Max) {
    m_ScrollMin = Min;
    m_ScrollMax = Max;
  
    if (m_ScrollPos < Min) {
      m_ScrollPos = Min;
    }
  
    if (m_ScrollPos > Max) {
      m_ScrollPos = Max;
    }

    return TRUE;
  }

  bool MTimeSlider::setScrollPos(int Pos) {
    if (m_ScrollPos != Pos) {
      m_ScrollPos = Pos;
  
      if (m_ScrollPos < m_ScrollMin) {
        m_ScrollPos = m_ScrollMin;
      }
  
      if (m_ScrollPos > m_ScrollMax) {
        m_ScrollPos = m_ScrollMax;
      }

      Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();
      scene->setTime(scene->frameToTick(m_ScrollPos));
      AztecViewManager::redrawAllViews();
      refresh();
    }

    return TRUE;
  }

  int MTimeSlider::getScrollPos() {
    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();
    return scene->tickToFrame(scene->getTime());
    return m_ScrollPos;
  }

  int MTimeSlider::getScrollMin() { 
    return m_ScrollMin; 
  }

  int MTimeSlider::getScrollMax() { 
    return m_ScrollMax; 
  }



#ifdef WIN32
  static int calcMajorTickSize(int width, int fromNum, int toNum, HDC dc) {
#else 
  static int calcMajorTickSize(int width, int fromNum, int toNum) {
#endif
    // start off with a tick size of 5.
    int result = 1;
    int lastEndPos;

    int numTicks;
    float tickStep;

    // Go through the ticks, and set an acceptable tick major value
  tryagain:
    {
      lastEndPos = 0;
    
      numTicks = toNum - fromNum + 1;
      tickStep = (float)width / numTicks;
    
      for (int n = 0; n < numTicks; n++)
      {
        int   xPos;
        std::string numStr;
      
        xPos = (int)(tickStep*n);
      
        if (n % result == 0)
        {
          // Draw the label text

          numStr = Aztec::MUIManager::tickCountToTimeStamp(Aztec::MScene::getGlobalScene()->frameToTick(n + fromNum));
        
          // Make sure that the new text position is in front
          // of the end of the last bit of text. If it does't over
          // lap, keep going, if it does, then increase our major tick size.
          if (xPos + 2 > lastEndPos) {
            int textWidth;
          
#ifdef WIN32
            SIZE  size;
            ::GetTextExtentPoint32(dc, numStr.c_str(), numStr.length(), &size);            
            textWidth = size.cx;
#else 
            // TODO: calculte this properly.
            textWidth = 32;
#endif
                      
            lastEndPos += textWidth + 8;
          } else {
            result += 5;
            result -= result % 5; 
            goto tryagain;
          }
        
        }
      
      }
    
    }
  

    return result;
  }


  bool MTimeSlider::onPaint() {
#ifdef WIN32

    HDC         dc;
    HBITMAP     bitmap, oldBitmap;

    RECT        ClientRect, Rect;
    HBRUSH      hBrush;
    HFONT       hOldFont;
    float       TickStep;
    int         n, NumTicks, LastTextEndPos, TickStart;
    int         TickMajor = 5;
  
    GetClientRect(m_hWnd, &ClientRect);
  
    dc = ::CreateCompatibleDC(paintDC);
    bitmap = ::CreateCompatibleBitmap(paintDC, ClientRect.right, ClientRect.bottom);
    oldBitmap = (HBITMAP)::SelectObject(dc, bitmap);
  
    hOldFont = (HFONT)::SelectObject(dc, GetStockObject(DEFAULT_GUI_FONT));
  
  
    {
      HPEN        hPen, hOldPen;
    
      hPen = CreatePen(PS_SOLID, 1, 0x000000);
      hOldPen = (HPEN)::SelectObject(dc, hPen);
      ::Rectangle(dc, 0,0,ClientRect.right, ClientRect.bottom);
      ::SelectObject(dc, hOldPen);
      DeleteObject(hPen);
    
      ClientRect.left += 1;
      ClientRect.top += 1;
      ClientRect.right  -= 1;
      ClientRect.bottom -= 1;
    }
  
    hBrush = ::CreateSolidBrush( RGB(255, 249, 204) );
    ::FillRect(dc, &ClientRect, hBrush);
    ::DeleteObject(hBrush);
  
    LastTextEndPos = 0;
  
    // Draw the Ticks
    ::SetBkMode(dc, TRANSPARENT);
  
    TickMajor = calcMajorTickSize(ClientRect.right - ClientRect.left, m_ScrollMin, m_ScrollMax, dc);
    NumTicks = m_ScrollMax - m_ScrollMin + 1;
    TickStart = ClientRect.left;
    TickStep = (float)(ClientRect.right-TickStart) / NumTicks;

    // Draw the tick marks.
    for (n = 0; n < NumTicks; n += TickMajor) {
      Rect.left = (int)(TickStep*n) + ClientRect.left;
      Rect.right = Rect.left + 2;
      Rect.top = ClientRect.top;
      Rect.bottom = ClientRect.bottom;

     ::FillRect(dc, &Rect, (HBRUSH)GetStockObject(BLACK_BRUSH));
    }

    // Draw the current cursor
    {
      hBrush = ::CreateSolidBrush(0xFFAA00);
    
      Rect.left = (int)(TickStep*(getScrollPos()-m_ScrollMin)) + TickStart;
      Rect.right = (int)(TickStep*((getScrollPos()-m_ScrollMin+1))) + TickStart;
      if (Rect.left == Rect.right) {
        Rect.right++;
      }
    
      Rect.top = ClientRect.top;
      Rect.bottom = ClientRect.bottom;
    
      ::FillRect(dc, &Rect, hBrush);
    
      ::DeleteObject(hBrush);
    }

    // Draw the Frame Numbers
    for (n = 0; n < NumTicks; n += TickMajor) {
      int xPos = (int)(TickStep*n) + ClientRect.left + 2;
      std::string numStr = Aztec::MUIManager::tickCountToTimeStamp(Aztec::MScene::getGlobalScene()->frameToTick(n+m_ScrollMin));
      
      ::TextOut(dc, xPos, 1, numStr.c_str(), numStr.length());

    }

    // Draw any of the keys in red marks
    Aztec::MBaseObjectPtr Obj;
    Aztec::MKeyableValuePtr keyableValue;
  
    KeyValueSet::iterator it;


    hBrush = ::CreateSolidBrush(0x0000FF);
      
    for (it = m_KeyingList.begin(); it != m_KeyingList.end(); ++it) {
      keyableValue = *it;
    
      std::vector<int> keyTimes;

      (*it)->getKeyTimes(keyTimes);
      for (unsigned n = 0; n < keyTimes.size(); ++n) {
        Aztec::MKeyPtr Key;
        int         Time;
        float       Pos;
      
        Time = keyTimes[n];
      
        Pos = (float)Time / 120;
      
        Rect.left = (int)(TickStep*Pos) + TickStart;
      
        Rect.right = (int)(TickStep*(Pos+1)) + TickStart;
        if (Rect.left == Rect.right) {
          Rect.right ++;
        }
      
        Rect.right = Rect.left+2;
      
        Rect.top = ClientRect.top;
        Rect.bottom = ClientRect.bottom;
      
        ::FillRect(dc, &Rect, hBrush);
      }
    }
  
    ::DeleteObject(hBrush);
    ::SelectObject(dc, hOldFont);
  
    GetClientRect(m_hWnd, &ClientRect);

    ::BitBlt(paintDC, 0,0,ClientRect.right, ClientRect.bottom, dc, 0,0,SRCCOPY);

    ::SelectObject(dc, oldBitmap);
    ::DeleteObject(bitmap);
    ::DeleteDC(dc);

    return true;

#else

    if (m_Handle != 0l) {
      // We don't want to waste cpu cycles on erasing
      // the background of this container widget every
      // time, so we're disabling it in here..
      static bool setbg = true;
      if (setbg) {
        m_Handle->setBackgroundMode(Qt::NoBackground);
        setbg = false;
      }

      QColorGroup cg = m_Handle->colorGroup();
      QRect rc = m_Handle->rect();
      QPainter p;

      // TODO: Only create buffer once @ init, resize when necessary
      QPixmap backbuf(rc.width(), rc.height());

      // Create the brushes, pens, etc.
      static QBrush cur(cg.highlight());//cg.mid());
      static QBrush emptybrush(Qt::NoBrush);
      static QPen emptyPen(Qt::NoPen);
      static QPen outlinecur(cg.button(), 1);
      static QPen tickmajor(cg.dark(), 2);
      static QPen tickminor(cg.mid(), 1);//button(), 1);

      backbuf.fill(cg.light());

      // Start painting our internal Qt widget
      //p.begin(m_Handle);
      p.begin(&backbuf);

      // Draw the ticks
      int tickMajor = calcMajorTickSize(rc.width() - rc.x(), m_ScrollMin, m_ScrollMax);
      int tickMinor = tickMajor / 5; // TODO: Compute correctly
      int numTicks = m_ScrollMax - m_ScrollMin + 1;
      int tickStart = rc.x();
      float tickStep = (float)(rc.width() - tickStart) / numTicks;


      // TODO: Use drawLineSegments() instead
      p.setPen(tickminor);
      for (int n = 0; n < numTicks; n += tickMinor) {
        p.drawLine((int)(tickStep*n) + rc.x(), rc.height() - 12, // TODO: Compute correctly
                   (int)(tickStep*n) + rc.x(), rc.height() - 6); //          "        "
      }

      p.setPen(tickmajor);
      for (int n = 0; n < numTicks; n += tickMajor) {
        p.drawLine((int)(tickStep*n) + rc.x(), rc.y()+20,//rc.y() + (rc.y()/3) /* tick = 1 quarter height */,
                   (int)(tickStep*n) + rc.x(), rc.height());
      }

      // Draw the current cursor
      p.setPen(outlinecur/*emptyPen*/);
      p.setBrush(cur);
      QRect rcur((int)(tickStep*(getScrollPos()-m_ScrollMin)) + tickStart, rc.y(), (int)(tickStep), rc.height());
      if (rcur.x() == rcur.width()) rcur.setWidth(rcur.width()+1);
      p.drawRect(rcur);


      // Draw frame numbers
      p.setPen(tickMajor);
      std::string strn;
      Aztec::MScenePtr ascene = Aztec::MScene::getGlobalScene();

      for (int n = 0; n < numTicks; n += tickMajor) {
        int xPos = (int)(tickStep*n) + rc.x() - 2;  // TODO: Compute center xcoord instead
        strn = Aztec::MUIManager::tickCountToTimeStamp(ascene->frameToTick(n + m_ScrollMin));
        p.drawText(xPos, rc.y()+12/*TODO: + fontheight */, strn.c_str());
      }


      // And finally draw the outer rectangle
      p.setPen(tickmajor);
      p.setBrush(emptybrush);
      p.drawRect(rc);

      p.end();

      // Blit
      bitBlt(m_Handle, 0,0, &backbuf, 0,0, backbuf.width(), backbuf.height());
    }

    return true;

#endif

  }

  bool MTimeSlider::onMousePressed(const Aztec::MMouseEvent &event) {
    Aztec::MSystemManager::getInstance()->getUndoManager()->beginUndo("Time change");
    int NumTicks, NewPos;
    float TickStep;
  
    Aztec::MRect2D clientRect = this->getClientSize();
  
    NumTicks = m_ScrollMax - m_ScrollMin;
    TickStep = (float)clientRect.getWidth() / NumTicks;
  
    NewPos = (int)((float)event.getX() / TickStep+0.5) + m_ScrollMin;
  
    setScrollPos(NewPos);

    m_Tracking = true;
    setMouseCapture(true);
  
    return true;
  }

  bool MTimeSlider::onMouseReleased(const Aztec::MMouseEvent &event) {
    if (m_Tracking) {
      m_Tracking = false;
      setMouseCapture(false);
    }

    Aztec::MSystemManager::getInstance()->getUndoManager()->endUndo();
    
    return true;
  }

  bool MTimeSlider::onMouseMove(const Aztec::MMouseEvent &event) {
    // TODO: Make platform independent!
#ifdef _WIN32
    if (m_Tracking) {
      int      NumTicks, NewPos;
      float    TickStep;
      RECT     ClientRect;
      
      ::GetClientRect(m_hWnd, &ClientRect);
      
      NumTicks = m_ScrollMax - m_ScrollMin;
      TickStep = (float)ClientRect.right / NumTicks;
      
      NewPos = (int)((float)(event.getX() / TickStep)+0.5) + m_ScrollMin;
      
      if (NewPos != getScrollPos()) {
        setScrollPos(NewPos);
      }
      return true;
    }

#else

    if (m_Tracking) {
      int numTicks, newPos;
      float tickStep;
      QRect rc = m_Handle->rect();

      numTicks = m_ScrollMax - m_ScrollMin;
      tickStep = (float)rc.width() / numTicks;

      newPos = (int)((float)(event.getX() / tickStep)+0.5) + m_ScrollMin;
      if (newPos != getScrollPos()) {
        setScrollPos(newPos);
      }

      return true;
    }

#endif

    return false;
  }


  void MTimeSlider::clearAllKeyLists() {
     m_KeyingList.clear();
  }

  void MTimeSlider::addKeyableValue(const Aztec::MKeyableValuePtr &value) {
    if (value != NULL) {
      m_KeyingList.insert(value);
    }
  }

}
