#ifndef Aztec2_FunctionManager_Header
#define Aztec2_FunctionManager_Header

#include <vector>
#include <string>
#include <map>

namespace AztecGUI {

  /**
   * This is a convinient typedef to quickly define a vector of strings.
   */
  typedef std::vector<std::string> StringVector;

  /**
   * The function manager is responsible for mapping from text names, to 
   * actual C functions to perform self-contained operations in Aztec. 
   *
   * These operations could be creating a sphere,  opening the UV editor 
   * window, saving a file, quitting, or absolutely anything.
   *
   * Each function must have a name, for example "doWork". This function will 
   * be accessible via java script using the Scene.doWork() function call.
   *
   * Arguments are passed into each function in a vector of strings, and there 
   * is also a string return value. If a function fills out this return string, 
   * that is the value that JavaScript uses as the result. If this string is 
   * empty, then the javascript interpreter will use the integer return value 
   * from the function.
   */
  class FunctionManager {
  public:
    /// The type indicating success or failure in a function
    typedef enum { SUCCEED = 0, FAIL = 1} FunctionReturnType;

    /**
     * The type that functions must be of in order to be registerable with the FunctionManager.
     */
    typedef int FunctionType(const StringVector &args, std::string &result);

    /**
     * This gets the C function from the given name.
     */
    FunctionType* getFunction(const char *name);

    /**
     * This assigns the given function to be matched with the given name.
     */
    bool registerFunction(const char *name, FunctionType *function, const char *category = NULL, const char *friendlyName = NULL);

  private:
    typedef std::map<std::string, FunctionType*> FunctionMap;
    FunctionMap funcMap;
  };

}

#endif

