#INCLUDE "string.bi"

#DEFINE switchTIDY    &h01
#DEFINE switchREPLACE &h02

TYPE def_t ' definitions for material script file
  AS STRING * 32    symbol
  AS STRING * 64    value
END TYPE

TYPE tex_t ' texture list for material script file
  AS STRING * 30  file
  AS USHORT       material
END TYPE

TYPE mat_t ' material list for material script file
  AS STRING * 100 flags
  AS STRING * 22  path
  AS USHORT       count
  AS USHORT       firstTexture
  AS USHORT       numTextures
END TYPE

TYPE vec2_t ' 4 bytes per offset
  AS SHORT        x
  AS SHORT        y
END TYPE

TYPE vec2f_t ' 8 bytes per offset
  AS SINGLE       x
  AS SINGLE       y
END TYPE

TYPE vec3_t ' 6 bytes per coordinate
  AS SHORT        x
  AS SHORT        y
  AS SHORT        z
END TYPE

TYPE plane_t ' 128 bytes per plane
  AS vec3_t       coord(2)
  AS STRING * 80  texture
  AS vec2_t       offset
  AS SINGLE       rotation
  AS vec2f_t      scale
  AS UINTEGER     flagsCont
  AS UINTEGER     flagsSurf
  AS UINTEGER     light
END TYPE

TYPE brush_t ' 4 bytes per brush
  AS USHORT       numPlanes   ' number of planes used
  AS USHORT       firstPlane
END TYPE

TYPE field_t ' 64 bytes per field
  AS STRING * 16  key
  AS STRING * 48  value
END TYPE

TYPE entity_t ' 8 bytes per
  AS USHORT       numFields   ' number of fields used
  AS USHORT       numBrushes  ' number of brushes used
  AS USHORT       firstField
  AS USHORT       firstBrush
END TYPE

TYPE map_t
  AS UINTEGER     numEntities ' used entities
  AS UINTEGER     numPlanes   ' used planes
  AS UINTEGER     numBrushes  ' used brushes
  AS UINTEGER     numFields   ' used fields
  AS UINTEGER     maxEntities ' size of the stack (in entries)
  AS UINTEGER     maxPlanes   ' size of the stack (in entries)
  AS UINTEGER     maxBrushes  ' size of the stack (in entries)
  AS UINTEGER     maxFields   ' size of the stack (in entries)
  AS UINTEGER     size        ' total size of the structure (in bytes)
  AS entity_t PTR entities    ' all entities
  AS plane_t PTR  planes      ' all planes
  AS brush_t PTR  brushes     ' all brushes
  AS field_t PTR  fields      ' all fields
  DECLARE CONSTRUCTOR()
  DECLARE DESTRUCTOR()
  DECLARE SUB restart()
  DECLARE SUB destroy()
  DECLARE FUNCTION getField(entity AS INTEGER, key AS STRING) AS field_t PTR
  DECLARE SUB setField(entity AS INTEGER, key AS STRING, value AS STRING)
END TYPE

DECLARE FUNCTION loadMAP(filename AS STRING, map AS map_t PTR) AS INTEGER
DECLARE SUB saveMAP(filename AS STRING, map AS map_t PTR)
DECLARE FUNCTION tokenize(src AS STRING) AS INTEGER
DECLARE SUB SubtleCleanUp(map AS map_t PTR)
DECLARE SUB scriptUnpack(msg AS STRING, defs() AS def_t)
DECLARE SUB scriptLoad(filename AS STRING, texs() AS tex_t, mats() AS mat_t)
DECLARE SUB MaterialUpdate(map AS map_t PTR, texs() AS tex_t, mats() AS mat_t)
DECLARE SUB catchCmd(maps() AS STRING, defs() AS STRING, switches AS UBYTE)

''
'' manually (re)initialize map
''
PRIVATE SUB map_t.restart()
  this.destroy()

  this.maxEntities = 2048
  this.numEntities = 0
  this.entities = CALLOCATE(SIZEOF(TYPEOF(*this.entities)) * this.maxEntities)
  this.size = this.maxEntities * SIZEOF(TYPEOF(*this.entities))
  
  this.maxFields = 4096
  this.numFields = 0
  this.fields = CALLOCATE(SIZEOF(TYPEOF(*this.fields)) * this.maxfields)
  this.size += this.maxFields * SIZEOF(TYPEOF(*this.fields))
  
  this.maxBrushes = 8192
  this.numBrushes = 0
  this.brushes = CALLOCATE(SIZEOF(TYPEOF(*this.brushes)) * this.maxbrushes)
  this.size += this.maxBrushes * SIZEOF(TYPEOF(*this.brushes))
  
  this.maxPlanes = 32767
  this.numPlanes = 0
  this.planes = CALLOCATE(SIZEOF(TYPEOF(*this.planes)) * this.maxplanes)
  this.size += this.maxPlanes * SIZEOF(TYPEOF(*this.planes))
END SUB

''
'' initialize map structure
''
CONSTRUCTOR map_t()
  this.restart()
END CONSTRUCTOR

''
'' destroy map structure (just in case we forgot to do it manually)
''
DESTRUCTOR map_t()
  this.destroy()
END DESTRUCTOR

''
'' destroy map structure
''
SUB map_t.destroy()
  IF (this.entities) THEN DEALLOCATE(this.entities): this.entities = 0
  this.numEntities = 0: this.maxEntities = 0

  IF (this.planes) THEN DEALLOCATE(this.planes): this.planes = 0
  this.numPlanes = 0: this.maxPlanes = 0
 
  IF (this.brushes) THEN DEALLOCATE(this.brushes): this.brushes = 0
  this.numBrushes = 0: this.maxBrushes = 0

  IF (this.fields) THEN DEALLOCATE(this.fields): this.fields = 0
  this.numFields = 0: this.maxFields = 0
END SUB

''
'' Get field according to key
''
PRIVATE FUNCTION map_t.getField(entity AS INTEGER, key AS STRING) AS field_t PTR
  DIM AS entity_t PTR ent

  IF ((entity < 0) OR (entity >= this.numEntities)) THEN RETURN 0
  
  ent = @this.entities[entity]
  FOR i AS UINTEGER = ent->firstField TO ent->firstField + ent->numFields - 1
    IF (RTRIM(this.fields[i].key) = key) THEN RETURN @this.fields[i]
  NEXT i

  RETURN 0
END FUNCTION

''
'' Insert field
''
PRIVATE SUB map_t.setField(entity AS INTEGER, key AS STRING, value AS STRING)
  DIM AS field_t PTR fld 

  IF ((entity < 0) OR (entity >= this.numEntities)) THEN EXIT SUB

  ' if the key already exists, modify and bail
  fld = this.getField(entity, key)
  IF (fld) THEN
    fld->value = value
    EXIT SUB
  END IF

  ' the key doesn't exist, insert slot in the fields list
  IF (LEN(value) = 0) THEN EXIT SUB ' value is empty, ignore request
  this.numFields += 1
  FOR i AS UINTEGER = this.numFields TO this.entities[entity].firstField STEP - 1
    this.fields[i] = this.fields[i - 1]
  NEXT i
  fld = @this.fields[this.entities[entity].firstField]
  fld->key = key
  fld->value = value
  this.entities[entity].numFields += 1
  
  ' now we need to push the first field for every subsequent entities
  FOR i AS UINTEGER = entity + 1 TO this.numEntities - 1
    this.entities[i].firstField += 1
  NEXT i
END SUB

''
'' Modify materials
''
PRIVATE SUB MaterialUpdate(map AS map_t PTR, texs() AS tex_t, mats() AS mat_t)
  DIM AS STRING       path, file, temp
  DIM AS UBYTE PTR    xPtr
  DIM AS UINTEGER PTR iPtr
  DIM AS UINTEGER     offset
  DIM AS UINTEGER     value

  FOR p AS UINTEGER = 0 TO map->numPlanes - 1
    temp = TRIM(map->planes[p].texture)
    path = LEFT(temp, INSTRREV(temp, "/"))
    file = RIGHT(temp, LEN(temp) - LEN(path))
    
    FOR m AS UINTEGER = 0 TO UBOUND(mats)
      IF (path <> mats(m).path) THEN CONTINUE FOR
      
      FOR t AS UINTEGER = mats(m).firstTexture TO mats(m).firstTexture + mats(m).numTextures - 1
      'for t as uinteger = 0 to ubound(texs)
        'if (texs(t).material <> m) then continue for
        IF (TRIM(texs(t).file) <> file) THEN CONTINUE FOR
        ' we've got the texture and material for that plane
        xPtr = CPTR(UBYTE PTR, STRPTR(mats(m).flags))
        offset = 0
        FOR j AS UINTEGER = 0 TO mats(m).count - 1
          ' get bit value
          value = *CPTR(UINTEGER PTR, @xPtr[offset + 1])
          ' get target
          IF (xPtr[offset] AND &h1) THEN
            iPtr = @map->planes[p].flagsCont
          ELSE
            iPtr = @map->planes[p].flagsSurf
          END IF
          ' get mode
          SELECT CASE (xPtr[offset] AND &h6)
            CASE &h2
              *iPtr OR= value ' enable
            CASE &h4
              *iPtr XOR= (*iPtr AND value) ' disable
            CASE &h6
              *iPtr = value ' overwrite
          END SELECT
          ' move to next operation
          offset += 5
        NEXT j
      NEXT t
    NEXT m
  NEXT p
END SUB

''
'' Save structure to file
''
PRIVATE SUB saveMAP(filename AS STRING, map AS map_t PTR)
  DIM AS INTEGER ff = FREEFILE
  DIM AS entity_t PTR ent
  DIM AS brush_t PTR  brs
  DIM AS plane_t PTR  pln
  
  OPEN filename FOR OUTPUT AS #ff
  FOR e AS UINTEGER = 0 TO map->numEntities - 1
    PRINT #ff, "// entity "; e
    PRINT #ff, "{"
    ent = @map->entities[e]
    FOR f AS UINTEGER = ent->firstField TO ent->numFields + ent->firstField - 1
      IF (LEN(TRIM(map->fields[f].key)) ANDALSO LEN(TRIM(map->fields[f].value))) THEN
        PRINT #ff, CHR(34) + map->fields[f].key + CHR(34) + " " + CHR(34) + map->fields[f].value + CHR(34)
      END IF
    NEXT f
    FOR b AS UINTEGER = ent->firstBrush TO ent->numBrushes + ent->firstBrush - 1
      PRINT #ff, "// brush "; b - ent->firstBrush
      PRINT #ff, "{"
      brs = @map->brushes[b]
      FOR p AS UINTEGER = brs->firstPlane TO brs->numPlanes + brs->firstPlane - 1
        pln = @map->planes[p]
        PRINT #ff, "( " + TRIM(STR(pln->coord(0).x)) + " " + TRIM(STR(pln->coord(0).y)) + " " + TRIM(STR(pln->coord(0).z)) + " ) ";
        PRINT #ff, "( " + TRIM(STR(pln->coord(1).x)) + " " + TRIM(STR(pln->coord(1).y)) + " " + TRIM(STR(pln->coord(1).z)) + " ) ";
        PRINT #ff, "( " + TRIM(STR(pln->coord(2).x)) + " " + TRIM(STR(pln->coord(2).y)) + " " + TRIM(STR(pln->coord(2).z)) + " ) ";
        PRINT #ff, TRIM(pln->texture) + " " + TRIM(STR(pln->offset.x)) + " " + TRIM(STR(pln->offset.y)) + " ";
        PRINT #ff, TRIM(STR(pln->rotation)) + " " + TRIM(STR(pln->scale.x)) + " " + TRIM(STR(pln->scale.y)) + " ";
        PRINT #ff, TRIM(STR(pln->flagsCont)) + " " + TRIM(STR(pln->flagsSurf)) + " " + TRIM(STR(pln->light))
      NEXT p
      PRINT #ff, "}"
    NEXT b
    PRINT #ff, "}"
  NEXT e
  CLOSE #ff
END SUB

''
'' Load map into structure
''
PRIVATE FUNCTION loadMAP(filename AS STRING, map AS map_t PTR) AS INTEGER
  DIM AS INTEGER      ff = FREEFILE
  DIM AS STRING       ln
  DIM AS INTEGER      numScope = 0
  DIM AS INTEGER      numLines = 0
  DIM AS entity_t PTR ent
  DIM AS field_t PTR  fld
  DIM AS plane_t PTR  pln
  DIM AS brush_t PTR  brs
  
  PRINT "---- "; filename; " ----"
  IF (DIR(filename, &h24) = "") THEN RETURN -1

  OPEN filename FOR INPUT AS #ff
  DO
    ' read line
    numLines += 1
    INPUT #ff, ln
    ln = TRIM(ln)
    IF (LEN(ln) = 0) THEN CONTINUE DO
    IF (LEFT(ln, 2) = "//") THEN CONTINUE DO
  
    ' proper scope
    IF (ln = "{") THEN
      numScope += 1
      SELECT CASE numScope
      CASE 1
        ' new entity
        IF (map->numEntities = map->maxEntities) THEN
          PRINT "Entity stack exhausted at " + STR(map->numEntities) + ", line " + STR(numLines)
          loadMAP = -2
          EXIT DO
        END IF
        ' increase entity count
        map->numEntities += 1
        ' select entity
        ent = @map->entities[map->numEntities - 1]
        ' first field and first brush
        ent->firstField = map->numFields
        ent->firstBrush = map->numBrushes
      CASE 2
        ' new brush in current entity
        IF (map->numBrushes = map->maxBrushes) THEN
          PRINT "Brush stack exhausted at " + STR(map->numBrushes) + ", line " + STR(numLines)
          loadMAP = -2
          EXIT DO
        END IF
        ' increase brush count
        map->numBrushes += 1
        ent->numBrushes += 1
        ' select brush
        brs = @map->brushes[map->numBrushes - 1]
        ' first plane
        brs->firstPlane = map->numPlanes
      CASE ELSE
        PRINT "Open bracket out of scope, something went wrong!"
        loadMAP = -3
        EXIT DO
      END SELECT
      CONTINUE DO
    ELSEIF (ln = "}") THEN  
      numScope -= 1
      CONTINUE DO
    END IF
    
    ' append data
    SELECT CASE numScope
    CASE 1
      ' add field to current entity
      IF (map->numFields = map->maxFields) THEN
        PRINT "Field stack exhausted at " + STR(map->numFields) + ", line " + STR(numLines)
        loadMAP = -2
        EXIT DO
      END IF
      ' select field
      fld = @map->fields[map->numFields]
      ' store key and value
      fld->key = ln
      INPUT #ff, ln
      fld->value = ln
      ' move to next field, increase field count for current entity
      map->numFields += 1
      ent->numFields += 1
    CASE 2
      ' add plane to current brush
      IF (map->numPlanes = map->maxPlanes) THEN
        PRINT "Plane stack exhausted at " + STR(map->numPlanes) + ", line " + STR(numLines)
        loadMAP = -2
        EXIT DO
      END IF
      map->numPlanes += 1
      brs->numPlanes += 1
      pln = @map->planes[map->numPlanes - 1]
      ' get tokens
      DIM AS ZSTRING PTR   xPtr = ANY
      DIM AS UINTEGER      offset = 0
      DIM AS UINTEGER      numWords = ANY, cntWords = 0
      numWords = tokenize(ln)
      IF (numWords) THEN
        xPtr = CPTR(ZSTRING PTR, STRPTR(ln))
        DO
          ' map files don't always have all parameters for every line
          SELECT CASE cntWords
            CASE 0, 3, 6  ' coord x
              pln->coord(cntWords \ 3).x = VAL(xPtr[offset])
            CASE 1, 4, 7  ' coord y
              pln->coord(cntWords \ 3).y = VAL(xPtr[offset])
            CASE 2, 5, 8  ' coord z
              pln->coord(cntWords \ 3).z = VAL(xPtr[offset])
            CASE 9  ' texture
              pln->texture = LCASE(xPtr[offset])
            CASE 10 ' offset x
              pln->offset.x = VAL(xPtr[offset])
            CASE 11 ' offset y
              pln->offset.y = VAL(xPtr[offset])
            CASE 12 ' rotation
              DIM t AS SINGLE
              t = VAL(xPtr[offset])
              WHILE (t < 0)
                t += 360
              WEND
              pln->rotation = t
            CASE 13 ' scale x
              pln->scale.x = VAL(xPtr[offset])
            CASE 14 ' scale y
              pln->scale.y = VAL(xPtr[offset])
            CASE 15 ' content flag
              pln->flagsCont = VAL(xPtr[offset])
            CASE 16 ' surface flag
              pln->flagsSurf = VAL(xPtr[offset])
            CASE 17 ' light
              pln->light = VAL(xPtr[offset])
          END SELECT
          offset += LEN(xPtr[offset]) + 1 ' get next token
          cntWords += 1
        LOOP WHILE (cntWords < numWords)
      END IF
    CASE ELSE
      PRINT "LINE " + STR(numLines) + ": Out of scope (" + STR(numScope) + "), something went wrong!"
      loadMAP = -3
      EXIT DO
    END SELECT
  LOOP UNTIL EOF(ff)
  CLOSE #ff

  PRINT "numEntities "; map->numEntities; "/"; map->maxEntities
  PRINT "  numFields "; map->numFields; "/"; map->maxFields
  PRINT " numBrushes "; map->numBrushes; "/"; map->maxBrushes
  PRINT "  numPlanes "; map->numPlanes; "/"; map->maxPlanes
END FUNCTION

''
'' Cleans entity fields a bit
''
PRIVATE SUB SubtleCleanUp(map AS map_t PTR)
  DIM AS INTEGER     angle
  DIM AS field_t PTR fld

  ' round texture adjustement values (only few versions of QBSP support floating point values)
  FOR i AS UINTEGER = 0 TO map->numPlanes - 1
    map->planes[i].rotation = INT(map->planes[i].rotation)
  NEXT i

  FOR i AS UINTEGER = 0 TO map->numFields - 1
    SELECT CASE TRIM(map->fields[i].key)
      CASE "angle" ' make it a positive integer
        angle = VAL(map->fields[i].value)
        IF ((angle <> -1) AND (angle <> -2)) THEN ' special case: up and down
          WHILE (angle < 0)
            angle += 360
          WEND
          IF (angle = 0) THEN
            map->fields[i].key = ""
            map->fields[i].value = ""
          ELSE
            map->fields[i].value = TRIM(STR(angle))
          END IF
        END IF

      CASE "_color", "color", "fogval", "fogval2", "_sun_color" ' round RGB attributes to 2 decimals
        DIM AS ZSTRING PTR  xPtr = ANY
        DIM AS UINTEGER     offset = 0
        DIM AS STRING       tmp = ""

        tokenize(map->fields[i].value)
        xPtr = CPTR(ZSTRING PTR, STRPTR(map->fields[i].value))
        FOR i AS INTEGER = 0 TO 2
          tmp += FORMAT(VAL(xPtr[offset]), "0.00") + " "
          offset += LEN(xPtr[offset]) + 1
        NEXT i
        MID(tmp, 2, 1) = "."
        MID(tmp, 7, 1) = "."
        MID(tmp, 12, 1) = "."
        map->fields[i].value = TRIM(tmp)

      CASE "origin" ' make origin coordinates integer
        DIM AS ZSTRING PTR  xPtr = ANY
        DIM AS UINTEGER     offset = 0
        DIM AS STRING       tmp = ""
        
        tokenize(map->fields[i].value)
        xPtr = CPTR(ZSTRING PTR, STRPTR(map->fields[i].value))
        FOR i AS INTEGER = 0 TO 2
          tmp += STR(VAL(xPtr[offset])) + " "
          offset += LEN(xPtr[offset]) + 1
        NEXT i
        map->fields[i].value = TRIM(tmp)
    END SELECT
  NEXT i
END SUB

''
'' Tokenize string (replaces source string!)
''
PRIVATE FUNCTION tokenize(src AS STRING) AS INTEGER
  DIM AS UBYTE PTR  sPtr = ANY, tPtr = ANY
  DIM AS STRING     temp
  DIM AS INTEGER    jumped = -1, offset = 0, count = 0

  temp = STRING(LEN(src) + 1, 0)
  sPtr = CPTR(UBYTE PTR, STRPTR(src))
  tPtr = CPTR(UBYTE PTR, STRPTR(temp))

  FOR i AS INTEGER = 0 TO LEN(src) - 1
    SELECT CASE sPtr[i]
      CASE 40, 41, 32, 9 ' ( ) space tab
        IF (jumped = 0) THEN
          jumped = -1
          offset += 1
        END IF
      CASE ELSE ' any character
        jumped = 0
        tPtr[offset] = sPtr[i]
        offset += 1
    END SELECT
  NEXT i

  ' count words
  offset = 1
  DO
    IF (tPtr[offset] = 0) THEN
      IF (tPtr[offset - 1] = 0) THEN EXIT DO
      count += 1
    END IF
    offset += 1
  LOOP

  src = temp
  RETURN count
END FUNCTION


''
'' Expand defines for script file
''
SUB scriptUnpack(msg AS STRING, defs() AS def_t)
  DIM AS STRING tmp
  DIM AS INTEGER ofs = ANY, doAgain = -1
  
  ' replace symbols found in the string by their value
  IF (UBOUND(defs) <> -1) THEN
    DO
      doAgain =0
      FOR i AS UINTEGER = LBOUND(defs) TO UBOUND(defs)
        tmp = TRIM(defs(i).symbol)
        DO
          ofs = INSTR(msg, tmp)
          IF (ofs = 0) THEN EXIT DO
          msg = LEFT(msg, ofs - 1) + TRIM(defs(i).value) + RIGHT(msg, LEN(msg) - (ofs + LEN(tmp) - 1))
          doAgain = -1
        LOOP
      NEXT i
    LOOP WHILE doAgain ' recursive to make sure we got them all
  END IF

  msg = TRIM(msg)
END SUB

''
'' Load material script file
''
SUB scriptLoad(filename AS STRING, texs() AS tex_t, mats() AS mat_t)
  DIM AS STRING       msg, symbol, value, path
  DIM AS ZSTRING PTR  xPtr = ANY
  DIM AS UINTEGER     offset = ANY, numWords = ANY, cnt = 0
  DIM AS UINTEGER     pack1 = ANY, pack2 = ANY, bitval = ANY
  DIM AS INTEGER      ff = FREEFILE

  REDIM AS def_t defs()

  IF (DIR(filename, &h24) = "") THEN EXIT SUB

  OPEN filename FOR INPUT AS #ff
  DO
    LINE INPUT #ff, msg
    IF (LEN(msg) = 0) THEN CONTINUE DO
    IF (LEFT(msg, 1) = ";") THEN CONTINUE DO

    IF (LEFT(msg, 7) = "#DEFINE") THEN
      ' remove #DEFINE and expand symbols (including the possible define!)
      msg = TRIM(RIGHT(msg, LEN(msg) - 7))
      scriptUnpack(msg, defs())
      
      ' tokenize
      numWords = tokenize(msg)
      xPtr = CPTR(ZSTRING PTR, STRPTR(msg))
      offset = 0
      
      ' get symbol and values
      symbol = xPtr[offset]
      value = ""
      offset += LEN(xPtr[offset]) + 1
      FOR i AS UINTEGER = 1 TO numWords - 1
        value += xPtr[offset] + " "
        offset += LEN(xPtr[offset]) + 1
      NEXT i
      
      ' add to define list
      REDIM PRESERVE defs(0 TO UBOUND(defs) + 1)
      defs(UBOUND(defs)).symbol = symbol
      defs(UBOUND(defs)).value = TRIM(value)

    ELSEIF (LEFT(msg, 8) = "#DEFAULT") THEN
      ' remove #DEFAULT and expand symbols
      msg = TRIM(RIGHT(msg, LEN(msg) - 8))
      scriptUnpack(msg, defs())
      
      ' tokenize
      numWords = tokenize(msg)
      xPtr = CPTR(ZSTRING PTR, STRPTR(msg))
      offset = 0

      ' get path and flag operations
      path = xPtr[offset]
      offset += LEN(xPtr[offset]) + 1
      value = ""
      pack1 = 0
      pack2 = 0
      bitval = 0
      cnt = 0
      FOR i AS UINTEGER = 1 TO numWords - 1
        IF (xPtr[offset] = ":surf") THEN
          pack1 = &h0 ' is surface
        ELSEIF (xPtr[offset] = ":cont") THEN
          pack1 = &h1 ' is content
        ELSE
          SELECT CASE ASC(LEFT(xPtr[offset], 1))
          CASE 124 ' | enable
              pack2 = &h2
              offset += 1
            CASE 126 ' ~ disable
              pack2 = &h4
              offset += 1
            CASE ELSE ' replace
              pack2 = &h6
          END SELECT
          IF LEFT(xPtr[offset], 2) = "0x" THEN
            bitval = VAL("&h" + xPtr[offset + 2])
          ELSE
            bitval = VAL(xPtr[offset])
          END IF
          value += CHR(pack1 OR pack2) + MKI(bitval)
          cnt += 1
        END IF
        offset += LEN(xPtr[offset]) + 1
      NEXT i
      
      ' add to material list
      REDIM PRESERVE mats(0 TO UBOUND(mats) + 1)
      WITH mats(UBOUND(mats))
        .path = path
        .count = cnt
        .flags = value
        .firstTexture = UBOUND(texs) + 1
        .numTextures = 0
      END WITH

    ELSE
      IF LEN(msg) THEN
        ' tokenize
        numWords = tokenize(msg)
        xPtr = CPTR(ZSTRING PTR, STRPTR(msg))
        offset = 0
        
        ' add to texture list
        FOR i AS UINTEGER = 0 TO numWords - 1
          IF LEN(xPtr[offset]) THEN
            REDIM PRESERVE texs(0 TO UBOUND(texs) + 1)
            texs(UBOUND(texs)).file = xPtr[offset]
            texs(UBOUND(texs)).material = UBOUND(mats)
            mats(UBOUND(mats)).numTextures += 1
          END IF
          offset += LEN(xPtr[offset]) + 1
        NEXT i
      END IF
    END IF
  LOOP UNTIL EOF(ff)
  CLOSE #ff
END SUB

''
'' Catch command line parameters
''
SUB catchCmd(maps() AS STRING, defs() AS STRING, switches AS UBYTE)
  DIM AS INTEGER argCnt = 1
  DIM AS STRING  argStr = ""
  
  DO
    argStr = TRIM(UCASE(COMMAND(argCnt)))
    IF (LEN(argStr) = 0) THEN EXIT DO

    SELECT CASE RIGHT(argStr, LEN(argStr) - INSTRREV(argStr, "."))
      CASE "TXT"
        REDIM PRESERVE defs(0 TO UBOUND(defs) + 1)
        defs(UBOUND(defs)) = COMMAND(argCnt)
      CASE "MAP"
        REDIM PRESERVE maps(0 TO UBOUND(maps) + 1)
        maps(UBOUND(maps)) = COMMAND(argCnt)
      CASE ELSE
        IF (argStr = "/TIDY") THEN
          switches OR= switchTIDY
        ELSEIF (argStr = "/REPLACE") THEN
          switches OR= switchREPLACE
        END IF
    END SELECT

    argCnt += 1
  LOOP
END SUB

' ----------------------------------------------------------------------------------------------

    REDIM AS STRING mapFiles()  ' map files to modify
    REDIM AS STRING defFiles(0) ' material files to load
    DIM AS UBYTE    switches    ' options
    DIM AS map_t myMap          ' whole level
    REDIM AS tex_t texs()       ' texture list
    REDIM AS mat_t mats()       ' material list
    
    PRINT "MatCheck Alpha .1 - 2020 ACC": PRINT

    defFiles(0) = "default.txt"
    catchCmd(mapFiles(), defFiles(), switches)
    IF (UBOUND(mapFiles) = -1) THEN END
   
    PRINT "Loading material files..."
    FOR i AS UINTEGER = 0 TO UBOUND(defFiles)
      scriptLoad(defFiles(i), texs(), mats())
    NEXT i

    PRINT "Processing MAP files...": PRINT
    FOR j AS UINTEGER = 0 TO UBOUND(mapFiles)
      IF (loadMAP(mapFiles(j), @myMap) = 0) THEN
        IF (switches AND switchTIDY) THEN SubtleCleanUp(@myMap)
        MaterialUpdate(@myMap, texs(), mats())

        IF (switches AND switchREPLACE) THEN
          saveMAP(mapFiles(j), @myMap)
        ELSE
          saveMAP(mapFiles(j) + ".new", @myMap)
        END IF
      END IF
      myMap.restart()
    NEXT j

    myMap.destroy()
