unit Main;

{==============================================================================}
interface
{==============================================================================}

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ExtCtrls, ComCtrls, ToolWin, Menus, Grids, SortGrid,
  Buttons, MMSystem, ZipMstr, ShellApi;

type

  TApplyRuleFlag = (arAddToList,arDeleteFromList);

  TMainForm = class(TForm)
    MainMenu: TMainMenu;
    MMFile: TMenuItem;
    ToolBar: TToolBar;
    StatusBar: TStatusBar;
    PanelL: TPanel;
    PanelR: TPanel;
    PanelRB: TPanel;
    PanelRT: TPanel;
    PanelRTT: TPanel;
    SortGridFiles: TSortGrid;
    PanelRBR: TPanel;
    PanelRBL: TPanel;
    PanelLT: TPanel;
    PanelLB: TPanel;
    ListBoxMaps: TListBox;
    SpdBtnAdd: TSpeedButton;
    SpdBtnDelete: TSpeedButton;
    SpdBtnLoad: TSpeedButton;
    SpdBtnSave: TSpeedButton;
    SpdBtnClear: TSpeedButton;
    GBCheckFor: TGroupBox;
    CBSky: TCheckBox;
    CBSounds: TCheckBox;
    CBTextures: TCheckBox;
    GBInclude: TGroupBox;
    CBBsp: TCheckBox;
    CBMap: TCheckBox;
    Image: TImage;
    PanelRBLR: TPanel;
    PanelRBLL: TPanel;
    PanelRBLLT: TPanel;
    ListBoxInclude: TListBox;
    BtnAdd: TButton;
    BtnDelete: TButton;
    BtnLoad: TButton;
    BtnSave: TButton;
    BtnClear: TButton;
    MMOption: TMenuItem;
    MMhelp: TMenuItem;
    MMFileZip: TMenuItem;
    N1: TMenuItem;
    MMFileExit: TMenuItem;
    MMOptionDir: TMenuItem;
    MMOptionSfx: TMenuItem;
    MMHelpAbout: TMenuItem;
    OpenDialog: TOpenDialog;
    CBFileId: TCheckBox;
    SaveDialog: TSaveDialog;
    MMWindow: TMenuItem;
    MMWindow640: TMenuItem;
    MMWindow800: TMenuItem;
    MMWindow1024: TMenuItem;
    ZipMaster: TZipMaster;
    MMOptionAutoSfx: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure MMWindow1024Click(Sender: TObject);
    procedure MMWindow800Click(Sender: TObject);
    procedure MMWindow640Click(Sender: TObject);
    procedure MMOptionDirClick(Sender: TObject);
    procedure SpdBtnAddClick(Sender: TObject);
    procedure SpdBtnClearClick(Sender: TObject);
    procedure SpdBtnDeleteClick(Sender: TObject);
    procedure SpdBtnLoadClick(Sender: TObject);
    procedure SpdBtnSaveClick(Sender: TObject);
    procedure SortGridFilesSelectCell(Sender: TObject; ACol, ARow: Integer;
      var CanSelect: Boolean);
    procedure BtnAddClick(Sender: TObject);
    procedure BtnDeleteClick(Sender: TObject);
    procedure BtnClearClick(Sender: TObject);
    procedure BtnLoadClick(Sender: TObject);
    procedure BtnSaveClick(Sender: TObject);
    procedure CBCheckClick(Sender: TObject);
    procedure MMOptionSfxClick(Sender: TObject);
    procedure MMHelpAboutClick(Sender: TObject);
    procedure MMFileExitClick(Sender: TObject);
    procedure MMFileZipClick(Sender: TObject);
    procedure ZipMasterProgress(Sender: TObject; ProgrType: ProgressType;
      Filename: String; FileSize: Integer);
    procedure MMOptionAutoSfxClick(Sender: TObject);
    procedure BtnEditClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure CBFileIdClick(Sender: TObject);
  private
    { Private declarations }
      FAtom: ATOM;
      procedure ResizeForm(w: integer; h: integer);
      procedure InitControls;
      procedure LoadStockItems;
      procedure UpdateCustomList;
      procedure UpdateSortGridFiles;
      procedure UpdateFileMenu;
      procedure UpdateMapButtons;
      procedure UpdateIncludeButtons;
      procedure LoadBsp(FileName: string);
      procedure LoadMap(FileName: string);
      procedure RemoveBlankLines(listbox: TListBox);
      procedure UpdateImagePreview;
  public
    { Public declarations }
      procedure ApplyIncludeRule(rule: string; aType: TApplyRuleFlag);
      function IsIncludeRule(str: string): Boolean;
  end;

  TPoint3f = packed record
    x,y,z: single;
  end;

  TPoint3s = packed record
    x,y,z: smallint;
  end;

  TBSPLump = packed record
    offset: longword;
    length: longword;
  end;

  TBSP = packed record
    magic: longword;
    version: longword;
    lump: packed array[0..18] of TBSPLump;
  end;

  TBSPTextInfo = packed record
    u_axis: TPoint3f;
    u_offset: single;
    v_axis: TPoint3f;
    v_offset: single;
    flags: longword;
    value: longword;
    texture_name: packed array[0..31] of char;
    next_texinfo: longword;
  end;

   // Declare functions in NViewLib.dll
   //function NViewLibSetLanguage(Lang: PChar): bool; Stdcall; external 'NViewLib.dll';
   //procedure NViewLibSetCustomLang(pProgress,pError, pLoad, pErrLoad, pWarning : PChar); Stdcall; external 'NViewLib.dll';
   //function NViewLibSaveAsJPG(Quality:Integer; FileName: PChar):bool; Stdcall; external 'NViewLib.dll';
   //function Load_JPG(FileName : PChar; ShowProgress: BooLean):hbitmap; Stdcall; external 'NViewLib.dll';
   //function Load_GIF(FileName : PChar; ShowProgress: BooLean):hbitmap; Stdcall; external 'NViewLib.dll';
   function NViewLibLoad(FileName : PChar; ShowProgress: BooLean):hbitmap; Stdcall; external 'NViewLib.dll';

{==============================================================================}
var
{==============================================================================}
   MainForm: TMainForm;

   // Kingpin stock stuff
   KPSkies:     TStringList;     // Skys.txt
   KPSounds:    TStringList;     // Sounds.txt
   KPTextures:  TStringList;     // Textures.txt

   // Anything found in a map
   MapSky:        String;
   MapSounds:     TStringList;
   MapTextures:   TStringList;

   CustomList:    TStringList;   // Hold custom items ready to export

   MyDir: String;                // kpcmf working directory

   GameInstallDir,            // Prevents from reading kpcmf.ini every times
   DefaultMapDir,
   DefaultOutputDir: string;

{==============================================================================}
implementation
{==============================================================================}

uses
   Path,
   OpenMap,
   IniFiles,
   Update,
   AlsoInc,
   Sfx,
   About,
   Progress;


var
   LastSelected: string;

   TotalSize1,
   TotalProgress1,
   TotalSize2,
   TotalProgress2:   Int64;

{$R *.dfm}

{==============================================================================}
procedure TMainForm.FormCreate(Sender: TObject);
var
   MyIni: TIniFile;
begin

   // Prevent from a second instance
   if GlobalFindAtom('KPCMF_RUNNING') = 0 then
      FAtom := GlobalAddAtom('KPCMF_RUNNING')
   else
   begin
      // ShowMessage('KPCMF is already running.');
      Application.Terminate;
   end;

   MyDir := GetCurrentDir;

   // Initialize Controls
   InitControls;

   if not FileExists(MyDir+'\kpcmf.ini') then
      MMOptionDirClick(Self)
   else
   begin
      MyIni := TIniFile.Create(MyDir+'\kpcmf.ini');
      GameInstallDir := MyIni.ReadString('Path','GameInstallDir','');
      DefaultOutputDir := MyIni.ReadString('Path','DefaultOutputDir','');
      DefaultMapDir := MyIni.ReadString('Path','DefaultMapDir','');
      MyIni.Free;
   end;

   // Create & Initialize Objects
   KPSkies :=                 TStringList.Create;
   KPSounds :=                TStringList.Create;
   KPTextures :=              TStringList.Create;

   MapSounds :=               TStringList.Create;
   MapSounds.Duplicates :=    dupIgnore;
   MapSounds.Sorted :=        True;

   MapTextures :=             TStringList.Create;
   MapTextures.Duplicates :=  dupIgnore;
   MapTextures.Sorted :=      True;

   CustomList :=              TStringList.Create;
   CustomList.Duplicates :=   dupIgnore;
   CustomList.Sorted :=       True;

   // Read KP Stock Items
   LoadStockItems;
end;

{==============================================================================}
procedure TMainForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
   KPSkies.Free;
   KPSounds.Free;
   KPTextures.Free;
   MapSounds.Free;
   MapTextures.Free;
   CustomList.Free;
end;

{==============================================================================}
procedure TMainForm.FormDestroy(Sender: TObject);
begin
   GlobalDeleteAtom(FAtom);
end;

{==============================================================================}
procedure TMainForm.LoadStockItems;
var
   fh: TextFile;
begin
  AssignFile(fh, MyDir+'\skys.txt');
  {$I-}
  Reset(fh);
  {$I+}
  if IOResult = 0 then begin
    KPSkies.LoadFromFile(MyDir+'\skys.txt');
    CloseFile(fh);
  end else
    MessageDlg('File skys.txt access error', mtWarning, [mbOk], 0);

  AssignFile(fh, MyDir+'\sounds.txt');
  {$I-}
  Reset(fh);
  {$I+}
  if IOResult = 0 then begin
    KPSounds.LoadFromFile(MyDir+'\sounds.txt');
    CloseFile(fh);
  end else
    MessageDlg('File sounds.txt access error', mtWarning, [mbOk], 0);

  AssignFile(fh, MyDir+'\textures.txt');
  {$I-}
  Reset(fh);
  {$I+}
  if IOResult = 0 then begin
    KPTextures.LoadFromFile(MyDir+'\textures.txt');
    CloseFile(fh);
  end else
    MessageDlg('File textures.txt access error', mtWarning, [mbOk], 0);
end;

{==============================================================================}
procedure TMainForm.InitControls;
begin
   with SortGridFiles do
   begin
      ColCount := 4;
      RowCount := 2;
      Cells[0,0] := 'File Name';
      Cells[1,0] := 'Type';
      Cells[2,0] := 'Directory';
      Cells[3,0] := 'Exists';
      ColWidths[0] := 110;
      ColWidths[1] := 48;
      ColWidths[2] := 230;
      ColWidths[3] := 2000;   // let it extend to the far right
   end;

   MMFileZip.Enabled :=    False;

   SpdBtnClear.Enabled :=  False;
   SpdBtnDelete.Enabled := False;
   SpdBtnSave.Enabled :=   False;

   BtnClear.Enabled :=     False;
   BtnDelete.Enabled :=    False;
   BtnSave.Enabled :=      False;

end;

{==============================================================================}
{ Resize and center main form                                                  }
{==============================================================================}
procedure TMainForm.ResizeForm(w: integer; h: integer);
var
   newLeft, newTop: integer;
begin
   newLeft := (Screen.Width - w) div 2;
   newTop  := (Screen.Height - h) div 2;
   // prevents form being four times repainted!
   SetBounds(newLeft, newTop, w, h);
end;

{==============================================================================}
procedure TMainForm.MMWindow1024Click(Sender: TObject);
begin
   ResizeForm(1024,768);
end;

{==============================================================================}
procedure TMainForm.MMWindow800Click(Sender: TObject);
begin
   ResizeForm(800,600);
end;

{==============================================================================}
procedure TMainForm.MMWindow640Click(Sender: TObject);
begin
   ResizeForm(640,480);
end;

{==============================================================================}
procedure TMainForm.MMOptionDirClick(Sender: TObject);
var
   FormPath: TFormPath;
begin
   FormPath := TFormPath.Create(self);
   try
      FormPath.ShowModal;
   finally
      FormPath.Free;
   end;
end;

{==============================================================================}
procedure TMainForm.LoadMAP(FileName: string);
var
   fh: TextFile;
   s,ss: string;
   i,j: integer;
begin
   AssignFile(fh, Filename);
   try
      Reset(fh);

      MapSounds.Clear;
      MapTextures.Clear;
      MapSky := '';

      while not Eof(fh) do begin
         readln(fh,s);

         // Ex. "sky" "k9fb"
         if MapSky = '' then
            if Pos('"sky"',s) <> 0 then
            begin
               i := 8;                       // Index of the 1st char of sky name
               while s[i] <> '"' do inc(i);  // Find out the last
               MapSky := Copy(s,8,i-8);
            end;

         // Ex. "noise" "k9fb/defense.wav"
         if Pos('"noise"',s) <> 0 then
         begin
            i := 10;
            while s[i] <> '"' do inc(i);
            MapSounds.Add(Copy(s,10,i-10));
         end;

         // Ex. ( 840 64 0 ) ( 840 192 0 ) ( 832 192 0 ) k9fb/fence1 0 0 0 1 1 0 0 0
         if s[1] = '(' then begin
            // Find the last ")"
            i := LastDelimiter(')',s);
            // Get to the 1st char of texture name
            inc(i,2);
            j := i;
            while s[j] <> ' ' do inc(j);
            ss := Copy(s,i,j-i);
            // Tricky part !!!
            if ss = 'test/num_0' then
            begin
               MapTextures.Add('test/num_0.tga');
               MapTextures.Add('test/num_1.tga');
               MapTextures.Add('test/num_2.tga');
               MapTextures.Add('test/num_3.tga');
               MapTextures.Add('test/num_4.tga');
               MapTextures.Add('test/num_5.tga');
               MapTextures.Add('test/num_6.tga');
               MapTextures.Add('test/num_7.tga');
               MapTextures.Add('test/num_8.tga');
               MapTextures.Add('test/num_9.tga');
               MapTextures.Add('test/num_colon.tga');
               MapTextures.Add('test/num_minus.tga');
               MapTextures.Add('test/num_space.tga');
            end
            else
               MapTextures.Add(ss+'.tga');
         end;

      end;  {*  while not Eof(fh) *}
      CloseFile(fh);
   except
      on EinOutError do MessageDlg('File I/O error.', mtError, [mbOk], 0);
   end;
end;

{==============================================================================}
{ Scan a .bsp file for custom sky, sounds and textures.                        }
{ MapSky, MapSounds are MapTextures are used to store the results              }
{==============================================================================}
procedure TMainForm.LoadBsp(FileName: string);
var
   fh: File;
   i,j,NumRead:  integer;
   bsp: TBSP;
   texinfo:      TBSPTextInfo;
   s: string;
   PtrEntity, ptr: ^char;
begin
   AssignFile(fh, FileName);
   try
      Reset(fh,1);  // record size = 1

      MapSounds.Clear;
      MapTextures.Clear;
      MapSky := '';

      BlockRead(fh, bsp, sizeof(TBSP), NumRead);     // Read BSP header

      if bsp.magic <> $50534249 then                 // "IBSP"
         MessageDlg('Invalid BSP file.', mtError, [mbOK], 0)
      else begin

         // Get texture names
         seek(fh, longint(bsp.lump[5].offset));      // Beginning of Texture Info Lump
         for i := 1 to (bsp.lump[5].length div sizeof(texinfo)) do
         begin
            BlockRead(fh, texinfo, sizeof(texinfo), NumRead);
            j := 0; s:='';
            while j < sizeof(texinfo.texture_name) do
            begin
               if texinfo.texture_name[j] = #0 then break;
               s := s + texinfo.texture_name[j];
               inc(j);
            end;
            MapTextures.Add(s+'.tga');
         end;

         // Get sky and sounds info
         seek(fh, longint(bsp.lump[0].offset));       // Beginning of Entities Lump
         GetMem(PtrEntity,bsp.lump[0].length);
         BlockRead(fh, PtrEntity^, bsp.lump[0].length, NumRead);
         ptr := PtrEntity;  s:='';

         for i := 1 to bsp.lump[0].length do
         begin
            // line break
            if ptr^ = #10 then
            begin
               // Check sky
               if MapSky = '' then
                  if Pos('"sky"',s) <> 0 then
                  begin
                     j := 8;
                     while s[j] <> '"' do inc(j);
                     MapSky := Copy(s,8,j-8);
                  end;
               // Check wav
               if Pos('"noise"',s) <> 0 then
               begin
                  j := 10;
                  while s[j] <> '"' do inc(j);
                  MapSounds.Add(Copy(s,10,j-10));
               end;

               s := '';      // reset s
               inc(ptr);     // advance to next string
            end
            else begin
               s := s + ptr^;
               inc(ptr);
            end;
         end;

         FreeMem(PtrEntity);

      end;  { if bsp.magic <> $50534249 }

      CloseFile(fh);
   except
      on EinOutError do MessageDlg('File I/O error.', mtError, [mbOk], 0);
   end;
end;

{==============================================================================}
procedure TMainForm.UpdateMapButtons;
var
   state: boolean;
begin

   if ListBoxMaps.Count = 0 then
      state := False
   else
      state := True;

   SpdBtnDelete.Enabled := state;
   SpdBtnSave.Enabled := state;
   SpdBtnClear.Enabled := state;
end;

{==============================================================================}
{ Sync SortGridFiles with CustomList                                           }
{==============================================================================}
procedure TMainForm.UpdateSortGridFiles;
var
   i: integer;
   s, FileName, FileDir, Base, Ext: string;
   SortOptions: TSortOptions;
begin
   UpdateFileMenu;

   if CustomList.Count = 0 then
   begin
      SortGridFiles.RowCount := 2;
      for i := 0 to 3 do SortGridFiles.Cells[i,1] := '';
   end
   else begin
      SortGridFiles.RowCount := CustomList.Count+1;

      FormUpdate.Caption := 'Updating Custom List';
      FormUpdate.Msg1.Caption := 'Updating file:';
      FormUpdate.ProgressBar.Max := CustomList.Count div 10;
      FormUpdate.Show;

      for i := 0 to CustomList.Count - 1 do
      begin

         FormUpdate.Msg2.Caption := CustomList.Strings[i];
         FormUpdate.ProgressBar.Position := i div 10;
         FormUpdate.Update;

         s := GameInstallDir+CustomList.Strings[i];
         FileName := StringReplace(s,'/','\',[rfReplaceAll]);
         FileDir := ExtractFileDir(FileName);
         Base := ExtractFileName(FileName);
         Ext := ExtractFileExt(FileName);
         Delete(Base,LastDelimiter('.',Base),Length(Ext));
         if Ext <> '' then Delete(Ext,1,1);

         SortGridFiles.Cells[0,i+1] := Base;
         SortGridFiles.Cells[1,i+1] := Ext;
         SortGridFiles.Cells[2,i+1] := FileDir;
         if FileExists(FileName) then
            SortGridFiles.Cells[3,i+1] := 'Yes'
         else
            SortGridFiles.Cells[3,i+1] := 'No';
      end;
      SortOptions.SortStyle := ssAlphabetic;
      SortOptions.SortDirection := sdAscending;
      SortOptions.SortCaseSensitive := True;
      SortGridFiles.SortByColumn( SortGridFiles.SortColumn, SortOptions);
      SortGridFiles.MoveTo(SortGridFiles.SortColumn,1);

      FormUpdate.Close;
      FormUpdate.Hide;
   end;
end;

{==============================================================================}
{ Scan each map file listed in ListBoxMaps for custom skies,sounds or textures }
{ CustomList (global) is used to keep those custom items.                      }
{==============================================================================}
procedure TMainForm.UpdateCustomList;
var
   i,j: integer;
   FileName: string;
   Base,Ext: string;
   s, missing: string;
begin
   CustomList.Clear;

   FormUpdate.Caption := 'Scanning Map Files';
   FormUpdate.Msg1.Caption := 'Scanning file:';
   FormUpdate.ProgressBar.Max := ListBoxMaps.Count;
   FormUpdate.Show;

   missing := '';
   for i := 0 to ListBoxMaps.Count - 1 do
   begin
      FormUpdate.Msg2.Caption := ListBoxMaps.Items.Strings[i];
      FormUpdate.ProgressBar.Position := i+1;
      FormUpdate.Update;

      FileName := DefaultMapDir + '/' + ListBoxMaps.Items.Strings[i];
      Base := ListBoxMaps.Items.Strings[i];
      Ext := ExtractFileExt(FileName);
      Delete(Base,LastDelimiter('.',Base),Length(Ext));  // get file base name

      if FileExists(FileName) then
      begin
         if CBBsp.Checked then CustomList.Add('/main/maps/'+Base+'.bsp');
         if CBMap.Checked then CustomList.Add('/main/maps/'+Base+'.map');
         if CBFileId.Checked then CustomList.Add('/main/txt/'+Base+'.txt');//FREDZ

         if UpperCase(Ext) = '.BSP' then LoadBsp(FileName);
         if UpperCase(Ext) = '.MAP' then LoadMap(FileName);

         if (MapSky <> '') and
            (KPSkies.IndexOf(LowerCase(MapSky)) = -1) and
            CBSky.Checked then
         begin
            CustomList.Add('/main/env/'+MapSky+'bk.tga');
            CustomList.Add('/main/env/'+MapSky+'dn.tga');
            CustomList.Add('/main/env/'+MapSky+'ft.tga');
            CustomList.Add('/main/env/'+MapSky+'lf.tga');
            CustomList.Add('/main/env/'+MapSky+'rt.tga');
            CustomList.Add('/main/env/'+MapSky+'up.tga');
			//FREDZ fix missing.
			CustomList.Add('/main/env/'+MapSky+'winrefl.tga');
         end;

         for j := 0 to MapTextures.Count-1 do
         begin
            if (KPTextures.indexOf(LowerCase(MapTextures.Strings[j])) = -1) and
               CBTextures.Checked then
            begin
               CustomList.Add('/main/textures/'+MapTextures.Strings[j]);
            end;
         end;

         for j := 0 to MapSounds.Count-1 do
         begin
            if Pos('.wav',LowerCase(MapSounds.Strings[j])) = 0 then   // append '.wav' if needed
               s := MapSounds.Strings[j] + '.wav'
            else
               s := MapSounds.Strings[j];
            if (KPSounds.IndexOf(LowerCase(s)) = -1) and
               CBSounds.checked then
            begin
               CustomList.Add('/main/sound/'+s);
            end;
         end;
      end
      else
      begin
         missing := missing + ListBoxMaps.Items.Strings[i] + ' ';
         ListBoxMaps.Selected[i] := True;
      end;
   end;

   FormUpdate.Close;
   FormUpdate.Hide;

   if missing <> '' then
   begin
      MessageDlg(missing+' not found and will be removed from map list.', mtInformation, [mbOK], 0);
      ListBoxMaps.DeleteSelected;
      ListBoxMaps.Update;
   end;

   for i := 0 to ListBoxInclude.Count-1 do
   begin
      if IsIncludeRule(ListBoxInclude.Items.Strings[i])
      then
         ApplyIncludeRule(ListBoxInclude.Items.Strings[i],arAddToList)
      else
         CustomList.Add(ListBoxInclude.Items.Strings[i]);
   end;

end;

{==============================================================================}
procedure TMainForm.UpdateFileMenu;
begin
   if CustomList.Count = 0 then
   begin
      MMFileZip.Enabled := False;
   end
   else begin
      MMFileZip.Enabled := True;
   end;
end;

{==============================================================================}
procedure TMainForm.RemoveBlankLines(listbox: TListBox);
var
   i: integer;
begin
   for i := 0 to listbox.Count -1 do
   begin
      if listbox.Items.Strings[i] = '' then
         listbox.Selected[i] := True;
   end;
   listbox.DeleteSelected;
end;

{==============================================================================}
procedure TMainForm.UpdateImagePreview;
begin
   if ListBoxMaps.Count = 0 then
      Image.Picture.Bitmap.Handle := HBITMAP(nil);
end;

{==============================================================================}
procedure TMainForm.SpdBtnAddClick(Sender: TObject);
var
   fn: string;
   i: integer;
begin
   OpenMapDialog.FileListBox.Directory := DefaultMapDir;

   if OpenMapDialog.ShowModal = mrOK then
   begin
      for i := 0 to OpenMapDialog.FileListBox.Count - 1 do
      begin
         if OpenMapDialog.FileListBox.Selected[i] then
         begin
            fn := ExtractFileName(OpenMapDialog.FileListBox.Items.Strings[i]);
            if ListBoxMaps.Items.IndexOf(fn) = -1 then ListBoxMaps.Items.Add(fn);
         end;
      end;
   end;

   UpdateMapButtons;
   UpdateCustomList;
   UpdateSortGridFiles;
end;

{==============================================================================}
procedure TMainForm.SpdBtnClearClick(Sender: TObject);
begin
   ListBoxMaps.Clear;
   UpdateMapButtons;
   UpdateCustomList;
   UpdateSortGridFiles;
   UpdateImagePreview;
end;

{==============================================================================}
procedure TMainForm.SpdBtnDeleteClick(Sender: TObject);
begin
   if ListBoxMaps.SelCount <> 0 then
   begin
      ListBoxMaps.DeleteSelected;
      UpdateMapButtons;
      UpdateCustomList;
      UpdateSortGridFiles;
      UpdateImagePreview;
   end;
end;

{==============================================================================}
procedure TMainForm.SpdBtnLoadClick(Sender: TObject);
var
   i: integer;
begin
   with OpenDialog do
   begin
      Title := 'Load Map List From...';
      InitialDir := MyDir;
      FileName := '';
      Filter := 'Map list files (*.kml)|*.kml';
      DefaultExt := 'kml';
      OptionsEx := OptionsEx + [ofExNoPlacesBar];
   end;

   if OpenDialog.Execute then
   begin
      if (ofExtensionDifferent in OpenDialog.Options) then
      begin
         MessageDlg('Loading a file with its extension name other than default ''kml'''+#13+#10+'is not suggested. You have been warned!', mtWarning, [mbOK], 0);
         OpenDialog.Options := OpenDialog.Options - [ofExtensionDifferent];
      end;
      ListBoxMaps.Items.LoadFromFile(OpenDialog.FileName);
      
      // Make it bullet-proof from bad list file format (mostly by advanced users)
      for i := 0 to ListBoxMaps.Count -1 do
         ListBoxMaps.Items.Strings[i] := Trim(ListBoxMaps.Items.Strings[i]);
      RemoveBlankLines(ListBoxMaps);

      UpdateMapButtons;
      UpdateCustomList;
      UpdateSortGridFiles;
      UpdateImagePreview;

   end;

end;

{==============================================================================}
procedure TMainForm.SpdBtnSaveClick(Sender: TObject);
begin
   with SaveDialog do
   begin
      Title := 'Save Map List To...';
      InitialDir := MyDir;
      Filter := 'Map list files (*.kml)|*.kml';
      DefaultExt := 'kml';
      Options := Options + [ofOverwritePrompt];
      OptionsEx := OptionsEx + [ofExNoPlacesBar];
   end;

   if SaveDialog.Execute then
      ListBoxMaps.Items.SaveToFile(SaveDialog.FileName);
end;

{==============================================================================}
procedure TMainForm.SortGridFilesSelectCell(Sender: TObject; ACol,
  ARow: Integer; var CanSelect: Boolean);
var
   s: string;
begin
   with SortGridFiles do begin

      s :=  Cells[2,ARow] +      // Directory
            '\' +
            Cells[0,ARow] +      // File base name
            '.' +
            Cells[1,ARow];       // File extension

      if LastSelected <> s then
      begin
         // if is wav file
         if UpperCase(Cells[1,ARow]) = 'WAV' then
         begin
            PlaySound(PChar(s),0,SND_FILENAME+SND_ASYNC);
            Image.Picture.Bitmap.Handle := HBITMAP(nil);
         end
         // Thanks to NViewLib!
         else begin
            PlaySound(nil,0,0);
            Image.Picture.Bitmap.Handle := NViewLibLoad(PChar(s), FALSE);
         end;
         LastSelected := s;
      end;
   end;
end;

{==============================================================================}
procedure TMainForm.UpdateIncludeButtons;
var
   state: boolean;
begin

   if ListBoxInclude.Count = 0 then
      state := False
   else
      state := True;

   BtnDelete.Enabled := state;
   BtnSave.Enabled := state;
   BtnClear.Enabled := state;
end;

{==============================================================================}
procedure TMainForm.BtnAddClick(Sender: TObject);
var
   FormInclude: TFormInclude;
begin
   FormInclude := TFormInclude.Create(self);
   FormInclude.Caption := 'Also Include...';
   FormInclude.DirectoryListBox.Directory := GameInstallDir;
   FormInclude.EditRule.Text := '/';
   FormInclude.MemoSorry.Visible := False;
   try
      if FormInclude.ShowModal = mrOK then
         UpdateSortGridFiles;
   finally
      FormInclude.Free;
   end;
   UpdateIncludeButtons;
end;

{==============================================================================}
procedure TMainForm.BtnDeleteClick(Sender: TObject);
var
   i: integer;
   s: string;
begin
   if ListBoxInclude.SelCount <> 0 then
   begin
      for i := 0 to ListBoxInclude.Count -1 do
      begin
         if ListBoxInclude.Selected[i] then
         begin
            s := ListBoxInclude.Items.Strings[i];
            if IsIncludeRule(s) then
               ApplyIncludeRule(s,arDeleteFromList)
            else begin
               if CustomList.IndexOf(s) <> -1 then
                  CustomList.Delete(CustomList.IndexOf(s));
            end;
         end;
      end;
      ListBoxInclude.DeleteSelected;
   end;
   UpdateIncludeButtons;
   UpdateSortGridFiles;
end;

{==============================================================================}
procedure TMainForm.BtnClearClick(Sender: TObject);
begin
   ListBoxInclude.SelectAll;
   BtnDeleteClick(self);
   UpdateIncludeButtons;
end;

{==============================================================================}
procedure TMainForm.BtnLoadClick(Sender: TObject);
var
   i: integer;
   s: string;
begin
   with OpenDialog do
   begin
      Title := 'Load Include List From...';
      InitialDir := MyDir;
      Filter := 'Include list files (*.kil)|*.kil';
      DefaultExt := 'kil';
      OptionsEx := OptionsEx + [ofExNoPlacesBar];
      FileName := '';
   end;

   if OpenDialog.Execute then
   begin
      if (ofExtensionDifferent in OpenDialog.Options) then
      begin
         MessageDlg('Loading a file with its extension name other than default ''kil'''+#13+#10+'is not suggested. You have been warned!', mtWarning, [mbOK], 0);
         OpenDialog.Options := OpenDialog.Options - [ofExtensionDifferent];
      end;

      // Clear old list
      if ListBoxInclude.Count <> 0 then
         BtnClearClick(Self);

      ListBoxInclude.Items.LoadFromFile(OpenDialog.FileName);

      // Make it bullet-proof from bad list file format
      for i := 0 to ListBoxInclude.Count -1 do
         ListBoxInclude.Items.Strings[i] := Trim(ListBoxInclude.Items.Strings[i]);
      RemoveBlankLines(ListBoxInclude);

      // Add include list into custom list
      for i := 0 to ListBoxInclude.Count - 1 do
      begin
         s := ListBoxInclude.Items.Strings[i];
         if IsIncludeRule(s) then
            ApplyIncludeRule(s,arAddToList)
         else
            CustomList.Add(s);
      end;

      UpdateSortGridFiles;

   end;

   UpdateIncludeButtons;

end;

{==============================================================================}
procedure TMainForm.BtnSaveClick(Sender: TObject);
begin
   with SaveDialog do
   begin
      Title := 'Save Include List To...';
      InitialDir := MyDir;
      Filter := 'Include list files (*.kil)|*.kil';
      DefaultExt := 'kil';
      Options := Options + [ofOverwritePrompt];
      OptionsEx := OptionsEx + [ofExNoPlacesBar];
   end;

   if SaveDialog.Execute then
      ListBoxInclude.Items.SaveToFile(SaveDialog.FileName);
end;

{==============================================================================}
function TMainForm.IsIncludeRule(str: string): Boolean;
var
   s: string;
begin
   s := StringReplace(str,'<MAP>','<MAP>',[rfIgnoreCase,rfReplaceAll]);
   if Pos('<MAP>',s) = 0 then
      Result := False
   else
      Result := True;
end;

{==============================================================================}
procedure TMainForm.ApplyIncludeRule(rule: string; aType: TApplyRuleFlag);
var
   i: integer;
   s, ss: string;
   base: string;
begin
   for i := 0 to ListBoxMaps.Count - 1 do
   begin
      s := rule;
      base := ListBoxMaps.Items.Strings[i];
      // Remove file extension including the dot. 100 should be safe
      // for extension size.
      Delete(base,LastDelimiter('.',base),100);
      ss := StringReplace(s,'<MAP>',base,[rfReplaceAll,rfignoreCase]);
      if aType = arAddToList then
      begin
         CustomList.Add(ss);
      end
      else
      begin
         // if not already in list
         if CustomList.IndexOf(ss) <> -1 then
            CustomList.Delete(CustomList.IndexOf(ss));
      end;
   end;
end;

{==============================================================================}
procedure TMainForm.CBCheckClick(Sender: TObject);
begin
   if ListBoxMaps.Count <> 0 then
   begin
      UpdateMapButtons;
      UpdateCustomList;
      UpdateSortGridFiles;
   end;
end;

{==============================================================================}
procedure TMainForm.MMOptionSfxClick(Sender: TObject);
var
  SFXForm : TSFXForm;
begin
  SFXForm := TSFXForm.Create(self);
  try
    SFXForm.ShowModal;
  finally
    SFXForm.free;
  end;
end;

{==============================================================================}
procedure TMainForm.MMHelpAboutClick(Sender: TObject);
var
  AboutForm : TAboutForm;
begin
  AboutForm := TAboutForm.Create(self);
  try
    AboutForm.ShowModal;
  finally
    AboutForm.free;
  end;
end;

{==============================================================================}
procedure TMainForm.MMFileExitClick(Sender: TObject);
begin
   Close;
end;

{==============================================================================}
procedure TMainForm.MMFileZipClick(Sender: TObject);
var
  MyIni : TIniFile;
  PathBuf: array[0..MAX_PATH] of Char;
  TempDir: string;
begin

   with SaveDialog do
   begin
      Title := 'Save to ...';
      Filter := 'Zip Files (*.zip)|*.zip';
      InitialDir := DefaultOutputDir;
      DefaultExt := 'zip';
      Options := SaveDialog.Options + [ofOverwritePrompt];
      OptionsEx := SaveDialog.OptionsEx + [ofExNoPlacesBar];
   end;

   if CBFileId.Checked then
   begin
      // Get Windows temp directory
      if Windows.GetTempPath(MAX_PATH, PathBuf) <> 0 then
      begin
         TempDir := PathBuf;
         // Removes any trailing '\'
         while (TempDir <> '') and (TempDir[Length(TempDir)] = '\') do
            TempDir := Copy(TempDir, 1, Length(TempDir) - 1);
      end
      else
         TempDir := MyDir;
     // FormFileId.MemoEdit.Lines.SaveToFile(TempDir+'\File_Id.diz');
   end;

   if SaveDialog.Execute then begin
      if FileExists(SaveDialog.FileName) then
         DeleteFile(SaveDialog.FileName);
      ProgressForm := TProgressForm.Create(self);
      StatusBar.SimpleText := 'Packing...';

      with ZipMaster do
      begin
         ZipFileName := SaveDialog.FileName;
         RootDir := GameInstallDir;
         AddCompLevel := 9;              // Compression level (fast) 0 - 9 (best)
         AddOptions := AddOptions + [AddDirNames];
         FSpecArgs.Assign(CustomList);

         // Disk Span Test
         //AddOptions := Zipmaster.AddOptions + [AddDiskSpan];
         //MaxVolumeSize := 1024*1024;       // 1mb

         ProgressForm.Show;
         Add;

         // Add File_Id.diz
         if CBFileId.Checked then
         begin
            AddOptions := AddOptions - [AddDirNames];
            FSpecArgs.Add(TempDir+'\File_Id.diz');
            Add;
            DeleteFile(TempDir+'\File_Id.diz');
         end;
      end;

      // Convert to SFX
      if MMOptionAutoSfx.Checked then
      begin
         MyIni := TIniFile.Create(MyDir+'\kpcmf.ini');
         ZipMaster.SFXCaption := MyIni.ReadString('SFX','Caption','Install Kingpin Map');
         if MyIni.ReadString('SFX','Message','<none>') <> '<none>' then
         begin
            ZipMaster.SFXMessage := MyIni.ReadString('SFX','Message','');
            if MyIni.ReadString('SFX','MessageType','Standard') = 'Information' then
               ZipMaster.SFXMessage := #1 + ZipMaster.SFXMessage;
            if MyIni.ReadString('SFX','MessageType','Standard') = 'Confirmation' then
               ZipMaster.SFXMessage := #2 + ZipMaster.SFXMessage;
         end;
         if MyIni.ReadString('SFX','CommandLine','<none>') <> '<none>' then
            ZipMaster.SFXCommandLine := MyIni.ReadString('SFX','CommandLine','');
         if MyIni.ReadString('SFX','OverwriteMode','Confirm') = 'Confirm' then
            ZipMaster.SFXOverWriteMode := OvrConfirm;
         if MyIni.ReadString('SFX','OverwriteMode','Confirm') = 'Always' then
            ZipMaster.SFXOverWriteMode := OvrAlways;
         if MyIni.ReadString('SFX','OverwriteMode','Confirm') = 'Never' then
            ZipMaster.SFXOverWriteMode := OvrNever;
         if MyIni.ReadString('SFX','Icon','sfx') <> 'sfx' then
         begin
            if MyIni.ReadString('SFX','Icon','sfx') = 'kpcmf' then
               ZipMaster.SFXIcon.LoadFromFile(MyDir+'\kpcmf.ico')
            else
               ZipMaster.SFXIcon.LoadFromFile(MyIni.ReadString('SFX','Icon',''));
         end;
         MyIni.Free;
         ZipMaster.ConvertSFX;
      end;

      ProgressForm.Close;
      ProgressForm.Free;
      StatusBar.SimpleText := '';
      if MMOptionAutoSfx.Checked then
         MessageDlg('Files have been successfully zipped and'+#13+#10+'converted to self-extractable.', mtInformation, [mbOK], 0)
      else
         MessageDlg('Files have been succssfully zipped.', mtInformation, [mbOK], 0);
   end;
end;

{==============================================================================}
{ Just a clip-n-paste from a demo source of Delphi Zip Component
{==============================================================================}
procedure TMainForm.ZipMasterProgress(Sender: TObject;
  ProgrType: ProgressType; Filename: String; FileSize: Integer);
var
   step : Integer;
begin
   case ProgrType of
      TotalSize2Process:
         begin
            // ZipMaster1Message( self, 0, 'in OnProgress type TotalBytes, size= ' + IntToStr( FileSize ) );
            ProgressForm.StatusBar1.Panels[0].Text := 'Total size: ' + IntToStr( FileSize div 1024 ) + ' Kb';
            ProgressForm.ProgressBar2.Position := 1;
            TotalSize2                    := FileSize;
            TotalProgress2                := 0;
         end;
      TotalFiles2Process:
         begin
            // ZipMaster1Message( self, 0, 'in OnProgress type TotalFiles, files= ' + IntToStr( FileSize ) );
            ProgressForm.StatusBar1.Panels[1].Text := IntToStr( FileSize ) + ' files';
         end;
      NewFile:
         begin
            // ZipMaster1Message( self, 0, 'in OnProgress type NewFile, size= ' + IntToStr( FileSize ) );
            ProgressForm.FileBeingZipped.Caption := Filename;
            ProgressForm.ProgressBar1.Position   := 1;         // Current position of bar.
            TotalSize1                      := FileSize;
            TotalProgress1                  := 0;
         end;
      ProgressUpdate:
         begin
            // ZipMaster1Message( self, 0, 'in OnProgress type Update, size= ' + IntToStr( FileSize ) );
            // FileSize gives now the bytes processed since the last call.
            TotalProgress1 := TotalProgress1 + FileSize;
            TotalProgress2 := TotalProgress2 + FileSize;
            if TotalSize1 <> 0 then
            begin
               {$IfDef VERD4+}  // D4+   (D5 gives a compiler error when using Int64 conversion!?)
               Step := MulDiv( TotalProgress1, 10000, TotalSize1 );
               {$Else}          // D2 and D3
               try
                  Step := Round( TotalProgress1 * 10000 / TotalSize1 );
               except
                  Step := 2147483647;
               end;
               {$EndIf}
               // ZipMaster1Message( self, 0, 'Step = ' + IntToStr( Step ) );
               ProgressForm.ProgressBar1.Position := 1 + Step;
            end else
               ProgressForm.ProgressBar1.Position := 10001;
            if TotalSize2 <> 0 then
            begin
               {$IfDef VERD4+}
               Step := MulDiv( TotalProgress2, 10000, TotalSize2 );
               {$Else}
               try
                  Step := Round( TotalProgress2 * 10000 / TotalSize2 );
               except
                  Step := 2147483647;
               end;
               {$EndIf}
               ProgressForm.ProgressBar2.Position := 1 + Step;
            end;
         end;
      EndOfBatch:    // Reset the progress bar and filename.
         begin
            // ZipMaster1Message( self, 0, 'in OnProgress type EndOfBatch' );
            ProgressForm.FileBeingZipped.Caption   := '';
            ProgressForm.ProgressBar1.Position     := 1;
            ProgressForm.StatusBar1.Panels[0].Text := '';
            ProgressForm.StatusBar1.Panels[1].Text := '';
            ProgressForm.ProgressBar2.Position     := 1;
         end;
   end;   // EOF Case
end;

{==============================================================================}
procedure TMainForm.MMOptionAutoSfxClick(Sender: TObject);
begin
   MMOptionAutoSfx.Checked := not MMOptionAutoSfx.Checked;
end;


{==============================================================================}
procedure TMainForm.BtnEditClick(Sender: TObject);
begin
//        ShellExecute(Handle, 'open', PChar('notepad'), PChar('/baseq2/txt/'+Base+'.txt'), nil, SW_SHOW);    //FREDZ todo
end;


procedure TMainForm.CBFileIdClick(Sender: TObject);
begin
   if ListBoxMaps.Count <> 0 then
   begin
      UpdateMapButtons;
      UpdateCustomList;
      UpdateSortGridFiles;
   end;
end;

end.


