// QUAKEMAPFILE.H
// Copyright  1999 Joe Riedel, Nick Randal.
// Author: Joe Riedel.
#ifndef QUAKEMAPFILE_H
#define QUAKEMAPFILE_H

#include "../TreadAPI.h"
#include <fstream.h>

//
// Reads/Writes Quake .map files. The CBaseTreadMapFile defines two routines: WriteMap() and ReadMap().
// Tread will invoke those procedures to, well, read and write maps of that type. Once those calls are
// made Tread has gone through the verification process, so reverifying type may be overkill. This code
// is straight out of Tread. We define other virtual calls here because it's very easy to "tweak" the file
// loading to make it work with Quake2 and HalfLife.
//
// The TREAD2XHEADERINFO structure defines information about the map being read/written, such as game type
// and the position of the different views. WriteMap() defines two lists that must be saved: the normal list
// and the selected object list. Tread doesn't consolidate things into a single list for map io. During reading
// you should use the document to attach objects to the map for the most part. The code in this class should give
// you a pretty good idea of how this is to be done. You can get away with linking object straight into the object
// list, and it often is faster to do this. Each call to AddObjectToMap() and AddObjectToSelection() causes Tread
// to updates certain class counts (it keeps tracks of the count of types of objects in the lists). Most of this
// code should make sense to those familiar with the Quake map format. For those not familiar with it, here's what it
// looks like:
//
// {
// "classname" "worldspawn"
// {	// brush 1
// ( x y z ) ( x y z ) ( x y z ) shiftx shifty scalex scaley rotation "texture"
// }
// }
//
// At least, I think that's what it is. That was off the top of my head, so forgive me if that wasn't
// quite right.
//
// Unfortunately, it's nearly impossible to tell a Quake map from a Quake2 map from a HalfLife map. You just really can't.
// Treads GDF2 scripts define a global definition called: def_gamemaptype. This allows map code to figure out if it should
// take control of reading/writing maps. 
//
// IMPORTANT NOTE: the hFile parameter passed on the ReadMap() / WriteMap() should *not* be closed. If you have to close
// the file (as we do in the code here) you must re-open it or Tread will die when you return from the call.
//

class CQuakeMapFile : public CBaseTreadMapFile
{
protected:

	int m_nEntityCount;
	int m_nBrushCount;
	int m_nFaceCount;

	virtual bool FindDuplicatePlane(CLinkedList<CQBrushPlane>* pPlaneList, CQBrushPlane* pPlane);

public:

	virtual bool WriteMap(CFile& hFile, TREAD2XHEADERINFO* pHeader, CTread3DDoc* pDocument, CLinkedList<CBaseObject>* pObjectList, CLinkedList<CBaseObject>* pSelectedObjectList, CLinkedList<CBaseObject>* pUIObjectList);
	virtual bool ReadMap(CFile& hFile, TREAD2XHEADERINFO* pHeader, CTread3DDoc* pDocument, CLinkedList<CBaseObject>* pObjectList, CLinkedList<CBaseObject>* pUIObjectList);

#define QRF_FATAL			-1
#define QRF_ERROR			0
#define QRF_OK				1
#define QRF_WORLDSPAWN		2
#define QRF_GROUP			3

	virtual int ReadEntity(CEntity** ppEntity, CTread3DDoc* pDocument, CTokenizer& Tokenizer);
	virtual int ReadBrush(CQBrush** ppBrush, CTread3DDoc* pDocument, CTokenizer& Tokenizer);
	virtual int ReadPlane(CQBrushPlane* pPlane, CTread3DDoc* pDocument, CTokenizer& Tokenizer);

	virtual bool WriteObject(CBaseObject* pObject, int nClass, CTread3DDoc* pDocument, fstream& file);
	virtual bool WriteObjectList(CLinkedList<CBaseObject>* pList, int nClass, CTread3DDoc* pDocument, fstream& file);
	virtual bool WriteGroup(CObjectGroup* pGroup, int nClass, CTread3DDoc* pDocument, fstream& file);
	virtual bool WriteWorldspawn(CEntity* pEntity, CTread3DDoc* pDocument, CLinkedList<CBaseObject>* pObjectList, CLinkedList<CBaseObject>* pSelectedObjectList, fstream& file);
	virtual bool WriteFace(CFace* pFace, CTread3DDoc* pDocument, fstream& file);
	virtual bool WriteBrush(CQBrush* pBrush, CTread3DDoc* pDocument, fstream& file);
	virtual bool WriteEntity(CEntity* pEntity, CTread3DDoc* pDocument, fstream& file);

};

#endif