/*
 Copyright (C) 2010-2017 Kristian Duske

 This file is part of TrenchBroom.

 TrenchBroom is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 TrenchBroom is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with TrenchBroom. If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include "FloatType.h"
#include "Model/NodeContents.h"
#include "NotifierConnection.h"
#include "View/MapDocument.h"

#include <vecmath/forward.h>

#include <map>
#include <memory>
#include <string>
#include <vector>

namespace TrenchBroom
{
namespace Model
{
enum class LockState;
enum class VisibilityState;
} // namespace Model

namespace View
{
class CommandProcessor;

/**
 * MapDocument API that is private to Command classes.
 *
 * These `performSomething()` methods will actually do an action, where
 * the corresponding `something()` in MapDocument would create and execute a
 * Command object which then calls `performSomething()`.
 */
class MapDocumentCommandFacade : public MapDocument
{
private:
  std::unique_ptr<CommandProcessor> m_commandProcessor;

  NotifierConnection m_notifierConnection;

public:
  static std::shared_ptr<MapDocument> newMapDocument();

private:
  MapDocumentCommandFacade();

public:
  ~MapDocumentCommandFacade() override;

public: // selection modification
  void performSelect(const std::vector<Model::Node*>& nodes);
  void performSelect(const std::vector<Model::BrushFaceHandle>& faces);
  void performSelectAllNodes();
  void performSelectAllBrushFaces();
  void performConvertToBrushFaceSelection();

  void performDeselect(const std::vector<Model::Node*>& nodes);
  void performDeselect(const std::vector<Model::BrushFaceHandle>& faces);
  void performDeselectAll();

public: // adding and removing nodes
  void performAddNodes(const std::map<Model::Node*, std::vector<Model::Node*>>& nodes);
  void performRemoveNodes(const std::map<Model::Node*, std::vector<Model::Node*>>& nodes);

  std::vector<std::pair<Model::Node*, std::vector<std::unique_ptr<Model::Node>>>>
  performReplaceChildren(
    std::vector<std::pair<Model::Node*, std::vector<std::unique_ptr<Model::Node>>>>
      nodes);

public: // swapping node contents
  void performSwapNodeContents(
    std::vector<std::pair<Model::Node*, Model::NodeContents>>& nodesToSwap);

public: // Node Visibility
  std::map<Model::Node*, Model::VisibilityState> setVisibilityState(
    const std::vector<Model::Node*>& nodes, Model::VisibilityState visibilityState);
  std::map<Model::Node*, Model::VisibilityState> setVisibilityEnsured(
    const std::vector<Model::Node*>& nodes);
  void restoreVisibilityState(
    const std::map<Model::Node*, Model::VisibilityState>& nodes);
  std::map<Model::Node*, Model::LockState> setLockState(
    const std::vector<Model::Node*>& nodes, Model::LockState lockState);
  void restoreLockState(const std::map<Model::Node*, Model::LockState>& nodes);

public: // layers
  using MapDocument::performSetCurrentLayer;

public:
  void performPushGroup(Model::GroupNode* group);
  void performPopGroup();

private:
  void doSetIssueHidden(const Model::Issue& issue, bool hidden) override;

public: // modification count
  void incModificationCount(size_t delta = 1);
  void decModificationCount(size_t delta = 1);

private: // notification
  void connectObservers();
  void documentWasNewed(MapDocument* document);
  void documentWasLoaded(MapDocument* document);

private: // implement MapDocument interface
  bool isCurrentDocumentStateObservable() const override;

  bool doCanUndoCommand() const override;
  bool doCanRedoCommand() const override;
  const std::string& doGetUndoCommandName() const override;
  const std::string& doGetRedoCommandName() const override;
  void doUndoCommand() override;
  void doRedoCommand() override;

  void doClearCommandProcessor() override;
  void doStartTransaction(std::string name, TransactionScope scope) override;
  void doCommitTransaction() override;
  void doRollbackTransaction() override;

  std::unique_ptr<CommandResult> doExecute(std::unique_ptr<Command> command) override;
  std::unique_ptr<CommandResult> doExecuteAndStore(
    std::unique_ptr<UndoableCommand> command) override;
};
} // namespace View
} // namespace TrenchBroom
